-- Loaded by clickabledata.lua
-- Defines behavior for clickable points in the cockpit
-- ie, momentary switch, dial, rotary knob...
--
-- The original functions are lifted from the A-4E mod
-- fcc_... functions are simplified or specialized versions prepared for this mod



-- These are the fields used in each entry of the elements table in clickabledata.lua, for reference.
--
-- Typically, you would generate these by using one of the button functions in clickable_defs.lua (filename may vary),
-- eg default_button() or springloaded_3_position_tumb(), but you can modify them manually or make your own button functions.
--- @class ClickableElement
--- @description Defines a clickable element's behavior, typically generated by functions in clickable_defs.lua
--- @field class table A table containing up to three class_type entries, corresponding to left-click, right-click, and scrolling interactions
--- @field hint string Text to be displayed when hovering over this element in the cockpit
--- @field device integer The index of the LUA device to connect to
--- @field action table<integer> Integers, A set of command codes to be sent to the device on left-click, right-click, and scroll actions.  MUST BE A UNIQUE INTEGER!
--- @field stop_action table<integer>? Integers, A set of command codes to be sent to the device on RELEASE of left-click or right-click, resetting the element to a neutral value.  Does not work with .LEV type.
--- @field arg table<integer|nil> Integers or nil, a set of animation args to connect this element to.  If set nil the element will not be able to remember its state!
--- @field arg_value table<number> Numbers, for class_type.BTN and .TUMB this is how much to increment per-click.  For .LEV it sets the starting value
--- @field arg_lim table<number> Numbers, this sets the range of animation of the element, and by extension the range of values the element can hold
--- @field cycle boolean? Defaults true.  If true, roll over upon exceeding a limit.  If false, stop
--- @field gain table<number>? Numbers, unsure what this does but I suspect it's to do with class_type.LEV sensitivity
--- @field use_release_message table<boolean>? Booleans, unsure what this does
--- @field sound table? Game sounds to play when interacting, exact formatting unknown
--- @field updatable boolean? Should this element's position be updated, ie is it animated?  Use sparingly
--- @field use_OBB boolean? Unknown, current theory is "use Oriented Bounding Box", which would be good for elements that rotate but don't translate during use.
--- @field animated table<boolean>? Booleans
--- @field animation_speed table<number>? Integers

cursor_mode =
{
	CUMODE_CLICKABLE = 0,
	CUMODE_CLICKABLE_AND_CAMERA = 1,
	CUMODE_CAMERA = 2
}

clickable_mode_initial_status = cursor_mode.CUMODE_CLICKABLE
use_pointer_name = true
anim_speed_default = 16

---Modified version of default_button.
---
---Suitable for buttons that initiate an action, this does not emit a signal on release.
---@param hint_ string
---@param device_ integer
---@param command_ integer
---@param animated_ boolean?
---@return table
function fcc_button(hint_, device_, command_, animated_)

	local animated_ = animated_ or false

	return {
		class         = {class_type.BTN},
		hint          = hint_,
		device        = device_,
		action        = {command_},
		arg           = {nil},
		arg_value     = {1},
		arg_lim       = {{0, 1}},
		use_OBB				= animated_,
		updatable			= animated_,
	}
end

---Modified version of default_button.
---
---Suitable for momentary buttons, this emits 1 on press and 0 on release.
---@param hint_ string
---@param device_ integer
---@param command_ integer
---@return table
function fcc_momentary_button(hint_, device_, command_)

	return {
		class         = {class_type.TUMB},
		hint          = hint_,
		device        = device_,
		action        = {command_},
		stop_action   = {command_},
		arg           = {nil},
		arg_value     = {1},
		arg_lim       = {{0, 1}},
	}
end

---Modified version of default_axis.
---
---A simple knob, can be click-dragged or interacted with using the scroll wheel.
---
---Scroll-up / Drag-right > 0, down / left < 0
---@param hint_ string
---@param device_ integer
---@param command_ integer
---@return table
function fcc_knob(hint_, device_, command_)

	return {
		class         = {class_type.LEV},
		hint          = hint_,
		device        = device_,
		action        = {command_},
		arg           = {nil},
		arg_value     = {0},
		arg_lim       = {{-1, 1}},
		gain          = {1},
		cycle         = false,
	}
end

---Useful for any rotary / multiposition switches, outputs 1 on left-click, -1 on right-click
---
---If on_release_ == true, also output 0 when the switch is released.  Useful for commands that require a second stopping-command.
---
---If animated_ == true, enable updating the connector position.
---@param hint_ string
---@param device_ integer
---@param command_ integer
---@param on_release_ boolean?
---@param animated_ boolean?
---@return table
function fcc_switch(hint_, device_, command_, on_release_, animated_)

	local animated_ = animated_ or false

	local stop_action_
	if on_release_ then
		stop_action_ = {command_, command_}
	else
		stop_action_ = nil
	end

	return {
		class         = {class_type.TUMB, class_type.TUMB},
		hint          = hint_,
		device        = device_,
		action        = {command_, command_},
		stop_action   = stop_action_,
		arg           = {nil, nil},
		arg_value     = {1, -1},
		arg_lim       = { {-1, 1}, {-1, 1} },
		cycle         = false,
		use_OBB				= animated_,
		updatable			= animated_,
	}
end

---Useful for any rotary / multiposition switches, outputs 1 on left-click, -1 on right-click, >0 on scroll-up, <0 on scroll-down.
---
---Does NOT support stop_action, since scrolling has no signal for it.
---@param hint_ string
---@param device_ integer
---@param command_ integer
---@return table
function fcc_switch_scrollable(hint_, device_, command_)

	return {
		class         = {class_type.TUMB, class_type.TUMB, class_type.LEV},
		hint          = hint_,
		device        = device_,
		action        = {command_, command_, command_},
		stop_action   = nil,
		arg           = {nil, nil, nil},
		arg_value     = {1, -1, 0},
		arg_lim       = { {-1, 1}, {-1, 1}, {-1, 1} },
		cycle         = false,
	}
end

function default_button(hint_, device_, command_, arg_, arg_val_, arg_lim_, sound_)
	local arg_val_ = arg_val_ or 1
	local arg_lim_ = arg_lim_ or {0, 1}

	return {
		class								= {class_type.BTN},
		hint								= hint_,
		device							= device_,
		action							= {command_},
		stop_action					= {command_},
		arg									= {arg_},
		arg_value						= {arg_val_},
		arg_lim							= {arg_lim_},
		use_release_message = {true},
		sound								= sound_ and {{sound_}, {sound_}} or nil
	}
end

-- not in use
function default_1_position_tumb(hint_, device_, command_, arg_, arg_val_, arg_lim_)
	local arg_val_ = arg_val_ or 1
	local arg_lim_ = arg_lim_ or {0, 1}
	return {
		class						= {class_type.TUMB},
		hint						= hint_,
		device					= device_,
		action					= {command_},
		arg							= {arg_},
		arg_value				= {arg_val_},
		arg_lim					= {arg_lim_},
		updatable				= true,
		use_OBB					= true
	}
end

function default_2_position_tumb(hint_, device_, command_, arg_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {1, -1},
		arg_lim         = {{0, 1}, {0, 1}},
		updatable       = true,
		use_OBB         = true,
		animated        = {true, true},
		animation_speed = {animation_speed_, animation_speed_},
		sound           = sound_ and {{sound_, sound_}} or nil
	}
end

function default_3_position_tumb(hint_, device_, command_, arg_, cycled_, inversed_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	local cycled = false

	local val = 1
	if inversed_ then
		val = -1
	end

	if cycled_ ~= nil then
		cycled = cycled_
	end

	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {val, -val},
		arg_lim         = {{-1, 1}, {-1, 1}},
		updatable       = true,
		use_OBB         = true,
		cycle           = cycled,
		animated        = {true, true},
		animation_speed = {animation_speed_, animation_speed_},
		sound           = sound_ and {{sound_, sound_}} or nil
	}
end

function inverted_3_position_tumb(hint_, device_, command_, arg_, cycled_, inversed_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	local cycled = false

	local val = 1
	if inversed_ then
		val = -1
	end

	if cycled_ ~= nil then
		cycled = cycled_
	end

	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {-val, val},
		arg_lim         = {{-1, 1}, {-1, 1}},
		updatable       = true,
		use_OBB         = true,
		cycle           = cycled,
		animated        = {true, true},
		animation_speed = {animation_speed_, animation_speed_},
		sound           = sound_ and {{sound_, sound_}} or nil
	}
end

function springloaded_forward_only_3_pos_tumb(hint_, device_, command_, arg_, inversed_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	local val = 1
	if inversed_ then
		val = -1
	end

	return {
		class               = {class_type.TUMB, class_type.TUMB},
		hint                = hint_,
		device              = device_,
		action              = {command_, command_},
		stop_action         = {nil, command_},
		arg                 = {arg_, arg_},
		arg_value           = {-val, val},
		arg_lim             = {{-1, 1}, {-1, 1}},
		updatable           = true,
		use_OBB             = true,
		cycle								= false,
		use_release_message = true,
		animated            = {true, true},
		animation_speed     = {animation_speed_, animation_speed_},
		sound               = sound_ and {{sound_, sound_}} or nil
	}
end

function springloaded_aft_only_3_pos_tumb(hint_, device_, command_, arg_, inversed_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	local val = 1
	if inversed_ then
		val = -1
	end

	return {
		class               = {class_type.TUMB, class_type.TUMB},
		hint                = hint_,
		device              = device_,
		action              = {command_, command_},
		stop_action         = {nil, command_},
		arg                 = {arg_, arg_},
		arg_value           = {val, -val},
		arg_lim             = {{-1, 1}, {-1, 1}},
		updatable           = true,
		use_OBB             = true,
		cycle								= false,
		use_release_message = true,
		animated            = {true, true},
		animation_speed     = {animation_speed_, animation_speed_},
		sound               = sound_ and {{sound_, sound_}} or nil
	}
end

-- This requires an arg to work, I think
function springloaded_3_pos_tumb(hint_, device_, command_, arg_, inversed_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	local val = 1
	if inversed_ then
		val = -1
	end

	return {
		class               = {class_type.BTN, class_type.BTN},
		hint                = hint_,
		device              = device_,
		action              = {command_, command_},
		stop_action         = {command_, command_},
		arg                 = {arg_, arg_},
		arg_value           = {val, -val},
		arg_lim             = {{-1, 1}, {-1, 1}},
		updatable           = true,
		use_OBB             = true,
		use_release_message = true,
		animated            = {true, true},
		animation_speed     = {animation_speed_, animation_speed_},
		sound               = sound_ and {{sound_, sound_}} or nil
	}
end

-- rotary axis with no end stops. suitable for continuously rotating knobs
function default_axis(hint_, device_, command_, arg_, default_, gain_, updatable_, relative_)
	local default = default_ or 1
	local gain = gain_ or 0.1
	local updatable = updatable_ or false
	local relative = relative_ or false

	return {
		class       = {class_type.LEV},
		hint        = hint_,
		device      = device_,
		action      = {command_},
		arg         = {arg_},
		arg_value   = {default},
		arg_lim     = {{0, 1}},
		updatable   = updatable,
		use_OBB     = true,
		gain        = {gain},
		relative    = {relative}
	}
end

function default_axis_limited(hint_, device_, command_, arg_, default_, gain_, updatable_, relative_, arg_lim_)
	local default = default_ or 0
	local updatable = updatable_ or false
	local relative = relative_ or false

	local gain = gain_ or 0.1
	return {
		class       = {class_type.LEV},
		hint        = hint_,
		device      = device_,
		action      = {command_},
		arg         = {arg_},
		arg_value   = {default},
		arg_lim     = {arg_lim_},
		updatable   = updatable,
		use_OBB     = false,
		gain        = {gain},
		relative    = {relative},
		cycle       = false
	}
end

function default_movable_axis(hint_, device_, command_, arg_, default_, gain_, updatable_, relative_)
	local default = default_ or 1
	local gain = gain_ or 0.1
	local updatable = updatable_ or false
	local relative = relative_ or false

	return {
		class			= {class_type.MOVABLE_LEV},
		hint			= hint_,
		device		= device_,
		action		= {command_},
		arg				= {arg_},
		arg_value	= {default},
		arg_lim		= {{0, 1}},
		updatable = updatable,
		use_OBB		= true,
		gain			= {gain},
		relative	= {relative}
	}
end

-- not in use. this multiple position switch is cyclable.
function multiposition_switch(hint_, device_, command_, arg_, count_, delta_, inversed_, min_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default

	local min_ = min_ or 0
	local delta_ = delta_ or 0.5

	local inversed = 1
	if inversed_ then
		inversed = -1
	end

	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {-delta_ * inversed, delta_ * inversed},
		arg_lim         = {
												{min_, min_ + delta_ * (count_ - 1)},
												{min_, min_ + delta_ * (count_ - 1)}
											},
		updatable       = true,
		use_OBB         = true,
		animated        = {true, true},
		animation_speed = {animation_speed_, animation_speed_},
		sound           = sound_ and {{sound_, sound_}} or nil
	}
end

function multiposition_switch_limited(hint_, device_, command_, arg_, count_, delta_, inversed_, min_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default

	local min_ = min_ or 0
	local delta_ = delta_ or 0.5

	local inversed = 1
	if inversed_ then
		inversed = -1
	end

	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {-delta_ * inversed, delta_ * inversed},
		arg_lim         = {
												{min_, min_ + delta_ * (count_ - 1)},
												{min_, min_ + delta_ * (count_ - 1)}
											},
		updatable       = true,
		use_OBB         = true,
		cycle           = false,
		animated        = {true, true},
		animation_speed = {animation_speed_, animation_speed_},
		sound           = sound_ and {{sound_, sound_}} or nil
	}
end

function multiposition_switch_limited_inverted(hint_, device_, command_, arg_, count_, delta_, inversed_, min_, sound_, animation_speed_)
	local animation_speed_ = animation_speed_ or anim_speed_default

	local min_ = min_ or 0
	local delta_ = delta_ or 0.5

	local inversed = 1
	if inversed_ then
		inversed = -1
	end

	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {delta_ * inversed, -delta_ * inversed},
		arg_lim         = {
												{min_, min_ + delta_ * (count_ - 1)},
												{min_, min_ + delta_ * (count_ - 1)}
											},
		updatable       = true,
		use_OBB         = true,
		cycle           = false,
		animated        = {true, true},
		animation_speed = {animation_speed_, animation_speed_},
		sound           = sound_ and {{sound_, sound_}} or nil
	}
end

-- rotary axis with push button
function default_button_axis(hint_, device_, command_1, command_2, arg_1, arg_2, limit_1, limit_2)
	local limit_1_ = limit_1 or 1.0
	local limit_2_ = limit_2 or 1.0
	return {
		class               = {class_type.BTN, class_type.LEV},
		hint                = hint_,
		device              = device_,
		action              = {command_1, command_2},
		stop_action         = {command_1, 0},
		arg                 = {arg_1, arg_2},
		arg_value           = {1, 0.5},
		arg_lim             = {{0, limit_1_}, {0, limit_2_}},
		animated            = {false, false},  --animated can make the knob wait for the animation to complete before another command is sent, leave this false otherwise big actions will make the movement strange.
		animation_speed     = {0, 0.4},
		gain                = {1.0, 0.1},
		relative            = {false, false},
		updatable           = true,
		use_OBB             = true,
		use_release_message = {true, false}
	}
end

-- NOT IN USE
function default_animated_lever(hint_, device_, command_, arg_, animation_speed_, arg_lim_)
	local animation_speed_ = animation_speed_ or anim_speed_default
	local arg_lim__ = arg_lim_ or {0.0, 1.0}
	return {
		class           = {class_type.TUMB, class_type.TUMB},
		hint            = hint_,
		device          = device_,
		action          = {command_, command_},
		arg             = {arg_, arg_},
		arg_value       = {1, 0},
		arg_lim         = {arg_lim__, arg_lim__},
		updatable       = true,
		gain            = {0.1, 0.1},
		animated        = {true, true},
		animation_speed = {animation_speed_, 0},
		cycle           = true
	}
end

function default_button_tumb(hint_, device_, command1_, command2_, arg_)
	return {
		class               = {class_type.BTN, class_type.TUMB},
		hint                = hint_,
		device              = device_,
		action              = {command1_, command2_},
		stop_action         = {command1_, 0},
		arg                 = {arg_, arg_},
		arg_value           = {-1, 1},
		arg_lim             = {{-1, 0}, {0, 1}},
		updatable           = true,
		use_OBB             = true,
		use_release_message = {true, false}
	}
end
