const stage_width = 1024;
const stage_height = 1024;

window.hideGui = function hideGui() {
    gui.hide();

    // Ensure buttons disappear properly
    const container = document.getElementById("imageSelectionContainer");
    if (container) {
        container.style.visibility = "hidden"; // Alternative method to hide
        container.style.display = "none"; // Ensures they fully disappear
    }
    setTimeout(() => {
        hb_send_proxy("close");
    }, 100);
}
window.showGui = function showGui() {
    gui.show();

    // Make image selection buttons visible again
    const container = document.getElementById("imageSelectionContainer");
    if (container) {
        container.style.visibility = "visible"; //  Restores visibility
        container.style.display = "block"; //  Ensures buttons fully reappear
    }
}
let brush_size = 0;
let selectedImageTexture = null;
let availableImages = [];
let currentTool = "brush";
let placedImages = [];

// GUI settings
const config = {
    draw_brush_size: 12,
    erase_brush_size: 50,
    brushColor: 0x000000,
    brushSmoothing: 0.5,
    useEraser: false,
    imageSize: 200, // Default image size for resizing
    imageOpacity: 1.0, //  New Opacity setting (1.0 = fully visible, 0.0 = invisible)
    imageRotation: 0
};

// PIXI App setup
const app = new PIXI.Application({
    backgroundColor: 0x40363a,
    backgroundAlpha: 0,
    resizeTo: window,
    antialias: true,
    forceCanvas: true,
});
document.body.appendChild(app.view);

// Canvas drawing layer setup
const renderTexture = PIXI.RenderTexture.create({ width: stage_width, height: stage_height });
const drawSprite = new PIXI.Sprite(renderTexture);
drawSprite.anchor.set(0.5);
drawSprite.position.set(stage_width / 2, stage_height / 2);
drawSprite.interactive = true;
drawSprite.eventMode = "static";
app.stage.addChild(drawSprite);

// Brush layer
const drawGraphics = new PIXI.Graphics();

// GUI setup
const gui = new dat.GUI();

const imageSelectionSettings = {
    selectedImage: null,
    placeImage: placeSelectedImage,
    paintImage: paintSelectedImageToCanvas,
};

const imageSelectionFolder = gui.addFolder("Image Selection");
imageSelectionFolder.open(); 

imageSelectionFolder.add(imageSelectionSettings, "paintImage").name("Paint to Canvas").onChange(() => {
    const guiButton = document.querySelector(".dg .property-name"); 
     if (guiButton) {
        guiButton.style.backgroundColor = "#FFFFFF"; 
        setTimeout(() => {
            guiButton.style.backgroundColor = ""; 
        }, 200);
    }
    paintSelectedImageToCanvas();
});


gui.add(config, 'draw_brush_size', 1, 100).name("Brush Size").onChange(updateBrush);
gui.add(config, 'erase_brush_size', 1, 200).name("Eraser Size").onChange(updateBrush);
gui.addColor(config, 'brushColor').name("Brush Color").onChange(updateBrush);
gui.add(config, "imageSize", 50, 1000).step(1).name("Resize Placed Image").onChange(updatePlacedImageSize);
gui.add(config, "imageOpacity", 0, 1).step(0.01).name("Image Opacity").onChange(updateImageOpacity);
gui.add(config, "imageRotation", -180, 180).step(1).name("Image Rotation").onChange(updateImageRotation);


// Function to update image size dynamically
function updateImageSize(value) {
    config.imageSize = value;
    console.log("New Image Size:", value);
}

function updateBrush() {
    brush_size = config.useEraser ? config.erase_brush_size : config.draw_brush_size;
}

// Drawing logic
let drawing = false;
let lastPosition = null;

drawSprite.on('mousedown', onPointerDown);
drawSprite.on('mousemove', onPointerMove);
drawSprite.on('mouseup', onPointerUp);
drawSprite.on('mouseupoutside', onPointerUp);
drawSprite.on('touchstart', onPointerDown);
drawSprite.on('touchmove', onPointerMove);
drawSprite.on('touchend', onPointerUp);

function onPointerDown(e) {
    if (currentTool !== "brush") return;

    config.useEraser = (e.data.originalEvent.button === 1); // Middle-click for eraser
    updateBrush();

    const pos = getRelativePos(e);
    drawing = true;
    lastPosition = pos;
    drawAt(pos);
}
function onPointerMove(e) {
    if (!drawing || currentTool !== "brush") return;

    const pos = getRelativePos(e);
    drawLine(lastPosition, pos);
    lastPosition = pos;
}
function onPointerUp() {
    drawing = false;
}
function getRelativePos(e) {
    const pos = drawSprite.toLocal(e.data.global);
    pos.x += stage_width / 2;
    pos.y += stage_height / 2;
    return pos;
}
function drawAt(pos) {
    drawGraphics.clear();

    if (config.useEraser) {
        drawGraphics.beginFill(0xFFFFFF, 1); // Force a strong erase stroke
        drawGraphics.blendMode = PIXI.BLEND_MODES.ERASE; // Ensures proper erasing
    } else {
        drawGraphics.beginFill(config.brushColor);
        drawGraphics.blendMode = PIXI.BLEND_MODES.NORMAL;
    }

    drawGraphics.drawCircle(pos.x, pos.y, brush_size / 2);
    drawGraphics.endFill();

    // Instead of clearing, we redraw onto the same texture
    app.renderer.render(drawGraphics, { renderTexture, clear: false });
}
function drawLine(from, to) {
    const dx = to.x - from.x;
    const dy = to.y - from.y;
    const distance = Math.sqrt(dx * dx + dy * dy);
    const steps = Math.ceil(distance / (brush_size / 4));

    for (let i = 0; i <= steps; i++) {
        const x = from.x + (dx * i) / steps;
        const y = from.y + (dy * i) / steps;
        drawAt({ x, y });
    }
}
// Image placement
function placeSelectedImage(imageName) {
    config.imageRotation = 0; // Reset rotation before placing new image
        gui.__controllers.forEach(controller => {
        if (controller.property === "imageRotation") {
            controller.setValue(0);
        }
    });
    const texture = PIXI.Texture.from("./Images/" + imageName);
    const sprite = new PIXI.Sprite(texture);
    sprite.anchor.set(0.5);
    sprite.position.set(stage_width / 2, stage_height / 2);
    sprite.rotation = (config.imageRotation * Math.PI) / 180;
    // Get original image dimensions
    const originalWidth = texture.baseTexture.width;
    const originalHeight = texture.baseTexture.height;
    // Calculate scaling factor based on `config.imageSize`
    const scaleFactor = config.imageSize / Math.max(originalWidth, originalHeight);
    // Apply scaling immediately
    sprite.width = originalWidth * scaleFactor; 
    sprite.height = originalHeight * scaleFactor;
    sprite.interactive = true;
    sprite.cursor = "move";
    sprite
        .on("pointerdown", onDragStart)
        .on("pointerup", onDragEnd)
        .on("pointerupoutside", onDragEnd)
        .on("pointermove", onDragMove);
    app.stage.addChild(sprite);
    placedImages.push(sprite);
    selectedImageTexture = texture;
    setTimeout(() => updatePlacedImageSize(config.imageSize), 50); //  Ensures resized settings apply correctly
}
function updatePlacedImageSize(value) {
    if (!placedImages.length) return; // No images placed yet

    placedImages.forEach(sprite => {
        const originalWidth = sprite.texture.baseTexture.width;
        const originalHeight = sprite.texture.baseTexture.height;
        const scaleFactor = value / Math.max(originalWidth, originalHeight);

        // Apply proportional scaling dynamically
        sprite.width = originalWidth * scaleFactor; 
        sprite.height = originalHeight * scaleFactor;
    });

    // Ensure images update visually after resizing
    app.renderer.render(app.stage);

    console.log("Updated all placed images dynamically to:", value);
}

function removePreviousImage() {
    if (!placedImages.length) return; // No image to remove
    const lastImage = placedImages.pop(); // Get last placed image
    app.stage.removeChild(lastImage); // Remove it from the stage
}
function onDragStart(event) {
    this.data = event.data;
    this.dragging = true;
    this.dragOffset = this.data.getLocalPosition(this);
    this.dragOffset.x *= this.scale.x;
    this.dragOffset.y *= this.scale.y;
}
function onDragEnd() {
    this.dragging = false;
    this.data = null;
    this.dragOffset = null;
}
function onDragMove() {
    if (this.dragging) {
        const newPosition = this.data.getLocalPosition(this.parent);
        this.position.x = newPosition.x - this.dragOffset.x;
        this.position.y = newPosition.y - this.dragOffset.y;
    }
}
function paintSelectedImageToCanvas() {
    if (!placedImages.length) return;
    const sprite = placedImages[placedImages.length - 1];
    const originalWidth = sprite.texture.baseTexture.width;
    const originalHeight = sprite.texture.baseTexture.height;
    const scaleFactor = config.imageSize / Math.max(originalWidth, originalHeight);
    const paintedWidth = originalWidth * scaleFactor;
    const paintedHeight = originalHeight * scaleFactor;
    sprite.width = paintedWidth;
    sprite.height = paintedHeight;
    sprite.alpha = config.imageOpacity;
    app.renderer.render(sprite, { renderTexture, clear: false });
    config.imageRotation = 0; // Reset rotation after painting
        gui.__controllers.forEach(controller => {
        if (controller.property === "imageRotation") {
            controller.setValue(0);
        }
    });
    sprite.width = paintedWidth;
    sprite.height = paintedHeight;
    app.stage.removeChild(sprite);
    placedImages.pop();
    console.log("Painted image while keeping aspect ratio and opacity:", paintedWidth, paintedHeight, "Alpha:", sprite.alpha);
}

function updateImageOpacity(value) {
    if (!placedImages.length) return; // No image placed yet

    const sprite = placedImages[placedImages.length - 1]; // Get last placed image
    sprite.alpha = value; //  Updates opacity

    console.log("Updated image opacity to:", value);
}
function updateImageRotation() {
    if (!placedImages.length) return;

    const sprite = placedImages[placedImages.length - 1];
    sprite.rotation = (config.imageRotation * Math.PI) / 180; //  Convert degrees to radians

    console.log("Updated rotation:", config.imageRotation);
}

// Function to populate dropdown dynamically
function updateImageSelectionUI(newImages) {
    const container = document.getElementById("imageSelectionContainer"); // Define it first
    container.innerHTML = ""; // Clear previous buttons
    container.style.display = "block"; // Restore visibility

    newImages.forEach(img => {
        const btn = document.createElement("button");
        btn.innerText = img.split("/").pop();
        btn.style.width = "100%";

        btn.onclick = () => {
        const lastPlacedImage = placedImages.length ? placedImages[placedImages.length - 1] : null;
    //  Prevent deselecting a painted image
        if (lastPlacedImage && lastPlacedImage.isPainted) {
            console.log("Skipping deselection—this image was painted to the canvas.");
        return; 
        }
    //  Normal de-selection behavior
        if (lastPlacedImage && lastPlacedImage.texture.baseTexture.resource.url.includes(img)) {
            app.stage.removeChild(lastPlacedImage);
            placedImages.pop();
            console.log("Deselected image:", img);
        } else {
            removePreviousImage();
            placeSelectedImage(img);
        }
};

        container.appendChild(btn);
    });

    console.log("Updated image selection UI with", newImages.length, "buttons.");
}

// Now that the function is defined, the fetch call can use it.
fetch("./Images/Images.json")
    .then(response => response.json())
    .then(data => {
        console.log("Loaded JSON data:", data);
        availableImages = data.images; 
        updateImageSelectionUI(availableImages); // Now calls the new button-based UI
    })
    .catch(error => console.error("Failed to load JSON:", error));