# Check if the script is already running with administrator privileges
if (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    # Restart the script with administrator privileges
    Start-Process powershell -ArgumentList "-File $PSCommandPath" -Verb RunAs
    exit
}

# Define the path of the modifications files
$modificationsFolder = Join-Path -Path $PSScriptRoot -ChildPath "Extended_Weapons\Modifications"
$logFilePath = Join-Path -Path $PSScriptRoot -ChildPath "Extended_Weapons\log.txt"

# Check if log file exists, if not create it and write initial message
if (-not (Test-Path $logFilePath)) {
    "Mod uninstalled" | Out-File -FilePath $logFilePath
}

# Function to apply modifications to the original file
function ApplyModifications {
    param(
        [string]$originalFilePath,
        [string]$searchTerm,
        [string]$modificationFilePath
    )

    Write-Host "Applying modifications from $modificationFilePath to $originalFilePath for search term $searchTerm"

    # Read the content of the modification file
    $modificationContent = Get-Content -Path $modificationFilePath
    Write-Host "Modification content read from $modificationFilePath"

    # Read the content of the original file
    $originalContent = Get-Content -Path $originalFilePath
    Write-Host "Original file content read from $originalFilePath"

    # Find the search term in the original content
    $insertionIndex = $null
    for ($i = 0; $i -lt $originalContent.Length; $i++) {
        if ($originalContent[$i] -match $searchTerm) {
            $insertionIndex = $i + 1
            break
        }
    }

    if ($insertionIndex -ne $null) {
        Write-Host "Search term '$searchTerm' found in $originalFilePath at line $insertionIndex"
        # Insert modification content after the search term
        $newContent = $originalContent[0..($insertionIndex-1)] + $modificationContent + $originalContent[$insertionIndex..($originalContent.Length - 1)]
        $newContent | Set-Content -Path $originalFilePath
        Write-Host "Modifications from $modificationFilePath applied to $originalFilePath"
    } else {
        Write-Host "Search term '$searchTerm' not found in $originalFilePath"
    }
}

# Function to handle Installation process
function InstallMod {
    if ((Get-Content -Path $logFilePath) -eq "Mod uninstalled") {
        foreach ($originalFile in $originalFiles.Keys) {
            Write-Host "Processing original file: $originalFile"
            $modificationFiles = $originalFiles[$originalFile]
            for ($i = 0; $i -lt $modificationFiles.Length; $i++) {
                $modificationFile = $modificationFiles[$i]
                $searchTerm = $searchTerms[$i]
                Write-Host "Processing modification file: $modificationFile with search term: $searchTerm"
                ApplyModifications -originalFilePath (Join-Path -Path $PSScriptRoot -ChildPath $originalFile) -searchTerm $searchTerm -modificationFilePath (Join-Path -Path $modificationsFolder -ChildPath $modificationFile)
            }
        }
        "Mod installed" | Out-File -FilePath $logFilePath
    } else {
        Write-Host "Mod already installed or log file is not set correctly."
    }
}

# Function to handle Uninstallation process
function UninstallMod {
    if ((Get-Content -Path $logFilePath) -eq "Mod installed") {
        foreach ($originalFile in $originalFiles.Keys) {
            Write-Host "Processing original file: $originalFile"
            $modificationFiles = $originalFiles[$originalFile]
            for ($i = 0; $i -lt $modificationFiles.Length; $i++) {
                $modificationFile = $modificationFiles[$i]
                $searchTerm = $searchTerms[$i]
                Write-Host "Processing modification file: $modificationFile with search term: $searchTerm"

                # Read the content of the original file
                $originalFilePath = Join-Path -Path $PSScriptRoot -ChildPath $originalFile
                $originalContent = Get-Content -Path $originalFilePath
                Write-Host "Original file content read from $originalFilePath"

                # Read the content of the modification file
                $modificationFilePath = Join-Path -Path $modificationsFolder -ChildPath $modificationFile
                $modificationContent = Get-Content -Path $modificationFilePath
                Write-Host "Modification content read from $modificationFilePath"

                # Remove the lines added by the modification file from the original file
                $modifiedContent = $originalContent | Where-Object { $_ -notin $modificationContent }
                $modifiedContent | Set-Content -Path $originalFilePath
                Write-Host "Modifications from $modificationFilePath removed from $originalFilePath"
            }
        }
        "Mod uninstalled" | Out-File -FilePath $logFilePath
    } else {
        Write-Host "Mod not installed or log file is not set correctly."
    }
}

# Declare the 6 original files with the 5 modifications file tied to them
$originalFiles = @{
    "VSN_F35A.lua" = @(
        "VSN_F35A_tips.lua",
        "VSN_F35A_outboardleft.lua",
        "VSN_F35A_inboard.lua",
        "VSN_F35A_fuselageRear.lua",
        "VSN_F35A_fuselageFront.lua"
    )
    "VSN_F35A_AG.lua" = @(
        "VSN_F35A_AG_tips.lua",
        "VSN_F35A_AG_outboardleft.lua",
        "VSN_F35A_AG_inboard.lua",
        "VSN_F35A_AG_fuselageRear.lua",
        "VSN_F35A_AG_fuselageFront.lua"
    )
    "VSN_F35B.lua" = @(
        "VSN_F35B_tips.lua",
        "VSN_F35B_outboardleft.lua",
        "VSN_F35B_inboard.lua",
        "VSN_F35B_fuselageRear.lua",
        "VSN_F35B_fuselageFront.lua"
    )
    "VSN_F35B_AG.lua" = @(
        "VSN_F35B_AG_tips.lua",
        "VSN_F35B_AG_outboardleft.lua",
        "VSN_F35B_AG_inboard.lua",
        "VSN_F35B_AG_fuselageRear.lua",
        "VSN_F35B_AG_fuselageFront.lua"
    )
    "VSN_F35C.lua" = @(
        "VSN_F35C_tips.lua",
        "VSN_F35C_outboardleft.lua",
        "VSN_F35C_inboard.lua",
        "VSN_F35C_fuselageRear.lua",
        "VSN_F35C_fuselageFront.lua"
    )
    "VSN_F35C_AG.lua" = @(
        "VSN_F35C_AG_tips.lua",
        "VSN_F35C_AG_outboardleft.lua",
        "VSN_F35C_AG_inboard.lua",
        "VSN_F35C_AG_fuselageRear.lua",
        "VSN_F35C_AG_fuselageFront.lua"
    )
}

# Declare all the 5 search terms
$searchTerms = @(
    "local tips",
    "local outboardleft",
    "local inboard",
    "local fuselageRear",
    "local fuselageFront"
)

# Prompt user to choose Installation or Uninstallation
Write-Host "Choose an option:"
Write-Host "1. Installation"
Write-Host "2. Uninstallation"

$keyPressed = Read-Host "Press 1 for Installation or 2 for Uninstallation"

if ($keyPressed -eq "1") {
    InstallMod
} elseif ($keyPressed -eq "2") {
    UninstallMod
} else {
    Write-Host "Invalid option selected."
}

# Prompt user to press Enter before closing the window
Read-Host "Press Enter to exit..."