-- Localizable strings
local LocStr = {
	APU68		= _("APU-68"),
	APU68UM3	= _("APU-68UM3"),
	O25			= _("O-25"),
	PodZones	= _("Pod Zones"),
	S24B		= _("S-24B - 240mm UnGd Rkt, 235kg, HE/Frag, (Low Smk)"),
	S25O		= _("S-25-O Fragmentation"),
	S25OFM		= _("S-25-OFM Hardened Target Penetrator"),
	Type		= _("Type"),
	UB32A		= _("UB-32A"),
	dNS			= {
		S5	= _("S-5"),
		S8	= _("S-8"),
		S13	= _("S-13"),
		S24	= _("S-24"),
		S25	= _("S-25"),
	}
}

-- Rocket launchers' parameters
local RLData = {
	B8M1 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 20,
		Cx_pil				= 0.00146484375,
		displayNamePrfx		= ("%s - 20 x %s, 80 %s"):format(_("B-8M1"), cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("B-20", "s-8_kom", 20, "tube_%d"),
		modelFileName		= "B-20",
		PictureBlendColor	= "0xffffffff",
		Picture				= "rus_B-8M1_S-8KOM.png",
		Weight_Empty		= 150,
	},
	B8V20A = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 20,
		Cx_pil				= 0.00213134765625,
		displayNamePrfx		= ("%s - 20 x %s, 80 %s"):format(_("B-8V20A"), cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("B-8V20A", "s-8_kom", 20, "tube_%d"),
		modelFileName		= "B-8V20A",
		Picture				= "rus_B-8V20A_S-8KOM.png",
		PictureBlendColor	= "0xffffffff",
		Weight_Empty		= 123,
	},
	B13L = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 5,
		Cx_pil				= 0.00159912109375,
		displayNamePrfx		= ("%s - 5 x %s, 122 %s"):format(_("B-13L"), cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("UB-13", "C-13", 5, "tube_%d"),
		modelFileName		= "UB-13",
		PictureBlendColor	= "0xffffffff",
		Picture				= "rus_B-13L_S-13OF.png",
		Weight_Empty		= 160,
	},
	BRU33 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "Point%02d",
		Count				= 2,
		Cx_pil				= 0.00244140625,
		displayNamePrfx		= "BRU-33: %d x",
		modelFileName		= "BRU_33A",
		Weight_Empty		= bru_33VER_mass,
	},
	LAU10 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%02d",
		Count				= 4,
		Cx_pil				= 0.001708984375,
		displayNamePrfx		= ("LAU-10 - 4 x %s, 127 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("LAU-10", "zuni", 4, "tube_%02d"),
		modelFileName		= "LAU-10",
		Picture				= "LAU10.png",
		Weight_Empty		= 61.7,
	},
	LAU61 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%02d",
		Count				= 19,
		Cx_pil				= 0.001708984375,
		displayNamePrfx		= ("LAU-61 - 19 x %s, 70 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("LAU-61", "hydra_m151he", 19, "tube_%02d"),
		modelFileName		= "LAU-61",
		Picture				= "LAU61.png",
		Weight_Empty		= 93, -- Arnold Defense spec sheet
	},
	LR25 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "Tube_%02d",
		Count				= 25,
		Cx_pil				= 0.00059912109375,
		displayNamePrfx		= ("LR-25 - 25 x %s, 50 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		modelFileName		= "LR-25",
		Picture				= "LR25.png",
		Weight_Empty		= 46,
		weapon_container_data = {
			nurs_args			= { 4 },
			launched_arg_state	= 1,
		}
	},
	M261 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 19,
		Cx_pil				= 0.001708984375,
		displayNamePrfx		= ("M261 - 19 x %s, 70 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("M261", "hydra_m151he", 19, "tube_%d"),
		modelFileName		= "M261",
		Picture				= "LAU61.png",
		Weight_Empty		= 36.3, -- Arnold Defense spec sheet
		genericWpnName		= "Hydra 70",
		zoneNumbersTable	= {
			["A"] = { 1, 2, 3, 4, 7, 8, 12, 13, 16, 17, 18, 19 },
			["B"] = { 5, 6, 14, 15 },
			["E"] = { 9, 10, 11 }
		},
		zoneRemapper		= function (zoneName) if zoneName == "C" then zoneName = "A" elseif zoneName == "D" then zoneName = "B" end return zoneName end
	},
	MATRAF1 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 36,
		Cx_pil				= 0.016 / 25,
		displayNamePrfx		= ("MATRA F1 - 36 x %s, 68 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("Matra-F1-Rocket", "SA342_SNEB68", 36),
		modelFileName		= "Matra-F1-Rocket",
		Picture				= "LAU61.png",
		Weight_Empty		= 88
	},
	MATRAF4 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 18,
		Cx_pil				= 0.007 / 25,
		displayNamePrfx		= ("MATRA F4 - 18 x %s, 68 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= {{ ShapeName = "M-2000C_LRF4", IsAdapter = true }},
		modelFileName		= "M-2000C_LRF4",
		Picture				= "LAU61.png",
		Weight_Empty		= 96.5
	},
	O25 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 1,
		Cx_pil				= 0.001708984375,
		DrawArgs			= {{ 3, 0.5 }},
		modelFileName		= "C-25PU",
		Weight_Empty		= 65,
	},
	Su27TwinPylon = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 2,
		Cx_pil				= 0.0004,
		displayNamePrfx		= "%d x",
		JettisonSubmunitionOnly	= true,
		modelFileName		= "su-27-twinpylon",
		Weight_Empty		= 32,
		elemGen				= function (CLSID) return {{ ShapeName = "su-27-twinpylon", IsAdapter = true }, { payload_CLSID = CLSID, connector_name = "S-25-L" }, { payload_CLSID = CLSID, connector_name = "S-25-R" }} end,
	},
	TELSON8 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		Count				= 8,
		Cx_pil				= 0.007,
		displayNamePrfx		= ("Telson 8 - 8 x %s, 68 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("SA342_Telson8", "SA342_SNEB68", 8),
		modelFileName		= "SA342_Telson8",
		Picture				= "SNEB68.png",
		Weight_Empty		= 30
	},
	TER9 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "Point%02d",
		Count				= 3,
		Cx_pil				= 0.00045,
		Cx_item				= -0.00001,
		displayNamePrfx		= "TER-9A: %d x",
		modelFileName		= "TER-9A",
		Weight_Empty		= ter_9a_mass,
	},
	UB16UMP = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 16,
		Cx_pil				= 0.00196533203125,
		displayNamePrfx		= ("%s - 16 x %s, 57 %s"):format(_("UB-16-57UMP"), cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("UB-16-57UMP", "S-5KO", 16, "tube_%d"),
		modelFileName		= "UB-16-57UMP",
		Picture				= "rus_UB-16_S-5KO.png",
		PictureBlendColor	= "0xffffffff",
		Weight_Empty		= 57,
	},
	UB32A = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 32,
		Cx_pil				= 0.00196533203125,
		displayNamePrfx		= ("%s - 32 x %s, 57 %s"):format(LocStr.UB32A, cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("UB_32A", "S-5KO", 32, "tube_%d"),
		modelFileName		= "UB_32A",
		Picture				= "rus_UB-32_S-5KO.png",
		PictureBlendColor	= "0xffffffff",
		Weight_Empty		= 106,
	},
	UB32A24 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 32,
		Cx_pil				= 0.00196533203125,
		displayNamePrfx		= ("%s-24 - 32 x %s, 57 %s"):format(LocStr.UB32A, cLocStr.UnGdRkts, cLocStr.mm),
		Elements			= ContainerWithSubshapes("UB_32A_24", "S-5KO", 32, "tube_%d"),
		modelFileName		= "UB_32A_24",
		Picture				= "rus_UB-32_S-5KO.png",
		PictureBlendColor	= "0xffffffff",
		Weight_Empty		= 106,
	},
	XM158 = {
		attribute			= {wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
		category			= CAT_ROCKETS,
		connNamePattern		= "tube_%d",
		Count				= 7,
		Cx_pil				= 0.00146484375,
		displayNamePrfx		= ("XM158 - 7 x %s, 70 %s"):format(cLocStr.UnGdRkts, cLocStr.mm),
		modelFileName		= "XM158",
		Picture				= "LAU68.png",
		Weight_Empty		= 17.7, -- AD815629, ENGINEERING FLIGHT TEST OF THE UH-1C HELICOPTER...
	},
}

--[[ --? Deprecated, superseded by BatchDeclareLoadout
rockets_data =
{
	["LAU-68_M151"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M151, HE", 			payload_CLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", mass = 113.9, mass_empty = 40.4, wsType = {4, 7, 33, 147}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK5"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk5, HEAT",			payload_CLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", mass = 102.1, mass_empty = 40.4, wsType = {4, 7, 33, 145}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_M156"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M156, Wht Phos", 	payload_CLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}", mass = 114.5, mass_empty = 40.4, wsType = {4,	7, 33, 148}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK61"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk61, Practice", 	payload_CLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}", mass = 104.2, mass_empty = 40.4, wsType = {4,	7, 33, 146}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_WTU-1/B"]	= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts WTU-1/B, Practice", payload_CLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}", mass = 113.2, mass_empty = 40.4, wsType = {4,	7, 33, 149}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK1"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk1, Practice",		payload_CLSID = "{FC85D2ED-501A-48ce-9863-49D468DDD5FC}", mass = 104.2, mass_empty = 40.4, wsType = {4, 7, 33, 144}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_M257"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M257, Para Illum",	payload_CLSID = "{647C5F26-BDD1-41e6-A371-8DE1E4CC0E94}", mass = 118.8, mass_empty = 40.4, wsType = {4, 7, 33, 151}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_M274"]		= {name = "LAU-68 pod - 7 x 2.75\" Hydra, UnGd Rkts M274, Practice Smk",payload_CLSID = "{0877B74B-5A00-4e61-BA8A-A56450BA9E27}", mass = 113.2, mass_empty = 40.4, wsType = {4, 7, 33, 150}, Cx = 0.00146484375, picture = "LAU68.png"},
	
	["LAU-131_M151"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M151, HE", 		payload_CLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}", mass = 102.3, mass_empty = 29.5, wsType = {4,	7, 33, 147}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_MK5"]		= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk5, HEAT", 		payload_CLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}", mass = 91.2, 	mass_empty = 29.5, wsType = {4,	7, 33, 145}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_M156"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M156, Wht Phos", 	payload_CLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}", mass = 103.6, mass_empty = 29.5, wsType = {4,	7, 33, 148}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_MK61"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk61, Practice", 	payload_CLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}", mass = 93.3, 	mass_empty = 29.5, wsType = {4,	7, 33, 146}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_WTU-1/B"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts WTU-1/B, Practice",payload_CLSID = "{DDCE7D70-5313-4181-8977-F11018681662}", mass = 102.3, mass_empty = 29.5, wsType = {4,	7, 33, 149}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_MK1"]		= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts Mk1, Practice",	payload_CLSID = "{D22C2D63-E5C9-4247-94FB-5E8F3DE22B71}", mass = 93.3, 	mass_empty = 29.5, wsType = {4, 7, 33, 144}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_M257"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M257, Para Illum",	payload_CLSID = "{DAD45FE5-CFF0-4a2b-99D4-5D044D3BC22F}", mass = 107.9, mass_empty = 29.5, wsType = {4, 7, 33, 151}, Cx = 0.00146484375, picture = "LAU131.png"},
	["LAU-131_M274"]	= {name = "LAU-131 pod - 7 x 2.75\" Hydra, UnGd Rkts M274, Practice Smk",payload_CLSID = "{6D6D5C07-2A90-4a68-9A74-C5D0CFFB05D9}",mass = 102.3,	mass_empty = 29.5, wsType = {4, 7, 33, 150}, Cx = 0.00146484375, picture = "LAU131.png"},
	
	["LAU-10"]			= {name = "LAU-10 pod - 4 x 127mm ZUNI, UnGd Rkts Mk71, HE/FRAG", 		payload_CLSID = "{F3EFE0AB-E91A-42D8-9CA2-B63C91ED570A}", mass = 288.9, mass_empty = 61.7, wsType = {4, 7, 33, 37},  Cx = 0.001708984375, picture = "LAU10.png"},
	["LAU-61"]			= {name = "LAU-61 pod - 19 x 2.75\" Hydra, UnGd Rkts M151, HE",			payload_CLSID = "{FD90A1DC-9147-49FA-BF56-CB83EF0BD32B}", mass = 273.4, mass_empty = 73.9, wsType = {4, 7, 33, 147}, Cx = 0.001708984375, picture = "LAU61.png"},
}

local function bru_33_2xlau(clsid,element,left,right)
	local lau_variant = rockets_data[element] or rockets_data["LAU-68_M151"]
	local ret = {
		category			=	CAT_ROCKETS,
		CLSID				=	clsid,
		Picture				=	lau_variant.picture,
		wsTypeOfWeapon		=	lau_variant.wsType,
		attribute			=	{4,	7,	32,	WSTYPE_PLACEHOLDER},
		Cx_pil				=	0.00244140625,
		Elements 			= {{ShapeName	=	"BRU_33A",IsAdapter  	   =   true  },}
	}
	if left then 	ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point02"}	end
	if right then	ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point01"}	end

	local sz = #ret.Elements - 1
	ret.Count  = sz
	ret.Weight_Empty = bru_33VER_mass + sz * lau_variant.mass_empty
	ret.Weight = bru_33VER_mass + sz * lau_variant.mass

	ret.Cx_pil = ret.Cx_pil + sz * lau_variant.Cx

	if sz > 1 then
		ret.displayName =	_("BRU-33").." with 2 x "..lau_variant.name
	else
		ret.displayName =	_("BRU-33").." with 1 x "..lau_variant.name
	end
	declare_loadout(ret)
end

bru_33_2xlau("{BRU33_2*LAU68}"	,	"LAU-68_M151",true,true)
bru_33_2xlau("{BRU33_LAU68}"	,	"LAU-68_M151",true, false)
bru_33_2xlau("{BRU33_LAU68_MK5}",	"LAU-68_MK5",true,false)
bru_33_2xlau("{BRU33_2*LAU68_MK5}",	"LAU-68_MK5",true,true)
bru_33_2xlau("{BRU33_LAU10}",		"LAU-10",true,false)
bru_33_2xlau("{BRU33_2*LAU10}",		"LAU-10",true,true)
bru_33_2xlau("{BRU33_LAU61}",		"LAU-61",true,false)
bru_33_2xlau("{BRU33_2*LAU61}",		"LAU-61",true,true)

-- For LAU-68/131
local function bru_42_ls_2x_lau_68_131(clsid,element,isleft)
	local lau_variant = rockets_data[element] or rockets_data["LAU-68_M151"]
	local ret = {
		category			=	CAT_ROCKETS,
		CLSID				=	clsid,
		Picture				=	lau_variant.picture,
		wsTypeOfWeapon		=	lau_variant.wsType,
		attribute			=	{4,	7,	32,	WSTYPE_PLACEHOLDER},
		Cx_pil				=	0.0005,
		Elements 			= 	{}
	}
	
	ret.Elements[#ret.Elements + 1] = { ShapeName	= "BRU-42_LS", IsAdapter  =   true}
	
	if isleft then	
		ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point03"}
	else
		ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point02"}	
	end
	ret.Elements[#ret.Elements + 1] = {payload_CLSID = lau_variant.payload_CLSID, connector_name =	"Point01"}

	local sz = #ret.Elements - 1
	ret.Count  = sz
	ret.Weight_Empty = 56.25 + sz * lau_variant.mass_empty
	ret.Weight = 56.25 + sz * lau_variant.mass
	ret.Cx_pil = ret.Cx_pil + sz * lau_variant.Cx
	ret.displayName =	_("BRU-42 with "..sz.." x "..lau_variant.name)
	
	declare_loadout(ret)
end

bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M151_L}", 	"LAU-68_M151", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M151_R}", 	"LAU-68_M151", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK5_L}", 	"LAU-68_MK5", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK5_R}", 	"LAU-68_MK5", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M156_L}", 	"LAU-68_M156", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M156_R}", 	"LAU-68_M156", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK61_L}", 	"LAU-68_MK61", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK61_R}", 	"LAU-68_MK61", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_WTU1B_L}", 	"LAU-68_WTU-1/B", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_WTU1B_R}", 	"LAU-68_WTU-1/B", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK1_L}", 	"LAU-68_MK1", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_MK1_R}", 	"LAU-68_MK1", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M257_L}", 	"LAU-68_M257", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M257_R}", 	"LAU-68_M257", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M274_L}", 	"LAU-68_M274", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU68_HYDRA_70_M274_R}", 	"LAU-68_M274", false)

bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M151_L}", 	"LAU-131_M151", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M151_R}", 	"LAU-131_M151", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK5_L}", 	"LAU-131_MK5", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK5_R}", 	"LAU-131_MK5", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M156_L}", 	"LAU-131_M156", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M156_R}", 	"LAU-131_M156", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK61_L}", 	"LAU-131_MK61", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK61_R}", 	"LAU-131_MK61", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_WTU1B_L}", 	"LAU-131_WTU-1/B", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_WTU1B_R}", 	"LAU-131_WTU-1/B", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK1_L}", 	"LAU-131_MK1", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_MK1_R}", 	"LAU-131_MK1", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M257_L}", 	"LAU-131_M257", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M257_R}", 	"LAU-131_M257", false)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M274_L}", 	"LAU-131_M274", true)
bru_42_ls_2x_lau_68_131("{BRU42LS_2*LAU131_HYDRA_70_M274_R}", 	"LAU-131_M274", false)

-- TER-9/A with 2 rockets
-- Because on one and on three modules have already been created before that
local function ter_9a_2rockets(element_)
	local rocket_variant = rockets_data[element_] or rockets_data["LAU-68_M151"]

	function createTableTER9(isLeft)
		local side_
		local secondConnector

		if isLeft then
			side_ = "L"
			secondConnector = "Point02"
		else
			side_ = "R"
			secondConnector = "Point03"
		end
		
		return	{
			category		=	CAT_ROCKETS,
			CLSID			=	"{TER_9A_2"..side_.."*"..element_.."}",
			Picture			=	rocket_variant.picture,
			wsTypeOfWeapon	=	rocket_variant.wsType,
			displayName		=  _("TER-9A with 2 x ")..rocket_variant.name,
			attribute		=	{wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER},
			Cx_pil			=	0.00045 + 2 * rocket_variant.Cx,
			Cx_item			= 	-0.00001,
			Count			=	14,
			Weight			=	ter_9a_mass + 2 * rocket_variant.mass,
			Elements		= 	{{ ShapeName = "TER-9A", IsAdapter = true },
								{ShapeName	= element_,	payload_CLSID = rocket_variant.payload_CLSID, connector_name =	"Point01"},
								{ShapeName	= element_,	payload_CLSID = rocket_variant.payload_CLSID, connector_name =	secondConnector}},

		}
	end

	declare_loadout(createTableTER9(true))
	declare_loadout(createTableTER9(false))
end

ter_9a_2rockets("LAU-68_MK5")		-- {TER_9A_2L*LAU-68_MK5} && {TER_9A_2R*LAU-68_MK5}
ter_9a_2rockets("LAU-68_M151")		-- {TER_9A_2L*LAU-68_M151} && {TER_9A_2R*LAU-68_M151}
ter_9a_2rockets("LAU-68_M156")		-- {TER_9A_2L*LAU-68_M156} && {TER_9A_2R*LAU-68_M156}
ter_9a_2rockets("LAU-68_MK61")		-- {TER_9A_2L*LAU-68_MK61} && {TER_9A_2R*LAU-68_MK61}
ter_9a_2rockets("LAU-68_WTU-1/B")	-- {TER_9A_2L*LAU-68_WTU-1/B} && {TER_9A_2R*LAU-68_WTU-1/B}

ter_9a_2rockets("LAU-131_MK5")		-- {TER_9A_2L*LAU-131_MK5} && {TER_9A_2R*LAU-131_MK5}
ter_9a_2rockets("LAU-131_M151")		-- {TER_9A_2L*LAU-131_M151} && {TER_9A_2R*LAU-131_M151}
ter_9a_2rockets("LAU-131_M156")		-- {TER_9A_2L*LAU-131_M156} && {TER_9A_2R*LAU-131_M156}
ter_9a_2rockets("LAU-131_MK61")		-- {TER_9A_2L*LAU-131_MK61} && {TER_9A_2R*LAU-131_MK61}
ter_9a_2rockets("LAU-131_WTU-1/B")	-- {TER_9A_2L*LAU-131_WTU-1/B} && {TER_9A_2R*LAU-131_WTU-1/B}
]]

---Fills the Elements table according to the launcher's zoning
---@param launcherData table Info about the launcher (see RLData)
---@param loadoutTable table Desired loadout for each zone
---@return table
local function fillZonedLauncher(launcherData, loadoutTable)
	local result = {}
	local tubeCLSIDPairs = {}

	result[#result + 1] = {
		ShapeName = launcherData.modelFileName,
		IsAdapter = true
	}

	for zoneName, CLSID in pairs(loadoutTable) do
		if launcherData.zoneRemapper then
			zoneName = launcherData.zoneRemapper(zoneName)
		end

		-- make CLSIDs correspond to tube numbers
		for __, tubeNumber in ipairs(launcherData.zoneNumbersTable[zoneName]) do
			tubeCLSIDPairs[tubeNumber] = CLSID
		end
	end

	for i, CLSID in ipairs(tubeCLSIDPairs) do
		result[i + 1] = {
			payload_CLSID = CLSID,
			connector_name = string.format(launcherData.connNamePattern, i)
		}
	end

	return result
end

local AGR_20_M151_UNGUIDED =
{
	category		= CAT_ROCKETS,
	name			= "AGR_20_M151_unguided",
	user_name		= _("APKWS M151 unguided"),
	display_name_short 	= "M151",
	scheme			= "nurs-standard",
	class_name		= "wAmmunitionNURS",
	model			= "AGR_20",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_NURS, wsType_Rocket, WSTYPE_PLACEHOLDER},

	warhead		= predefined_warhead("HYDRA_70_M151"),

	shape_table_data =
	{
		{
			name  = "AGR_20_M151_unguided";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS M151 unguided"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},

	engine = {
		fuel_mass   				= 7.5,
		impulse     				= 210,
		boost_time  				= 0,
		work_time   				= 1.8,
		boost_factor				= 1,
		nozzle_position 	   		= {{-0.95, 0, 0}},
		nozzle_orientationXYZ		= {{0, 0, 0}},
		tail_width  				= 0.052,
		boost_tail  				= 1,
		work_tail   				= 1,
		smoke_color					= {0.5, 0.474, 0.443},
		smoke_transparency			= 0.5,
	},

	fm = {
		mass        = 15,
		caliber     = 0.07,
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5,
		wind_time	= 1.8,
	},

	properties = {
		dist_min = 500,
		dist_max = 4000,
	},
}
declare_weapon(AGR_20_M151_UNGUIDED)

local AGR_20_M282_UNGUIDED =
{
	category		= CAT_ROCKETS,
	name			= "AGR_20_M282_unguided",
	user_name		= _("APKWS M282 unguided"),
	display_name_short 	= "M-282",
	scheme			= "nurs-standard",
	class_name		= "wAmmunitionNURS",
	model			= "AGR_20",
	wsTypeOfWeapon 	= {wsType_Weapon, wsType_NURS, wsType_Rocket, WSTYPE_PLACEHOLDER},

	warhead		= predefined_warhead("HYDRA_70_M282"),

	shape_table_data =
	{
		{
			name  = "AGR_20_M282_unguided";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS M282 unguided"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},

	engine = {
		fuel_mass   				= 7.5,
		impulse     				= 210,
		boost_time  				= 0,
		work_time   				= 1.8,
		boost_factor				= 1,
		nozzle_position 	   		= {{-0.95, 0, 0}},
		nozzle_orientationXYZ		= {{0, 0, 0}},
		tail_width  				= 0.052,
		boost_tail  				= 1,
		work_tail   				= 1,
		smoke_color					= {0.5, 0.474, 0.443},
		smoke_transparency			= 0.5,
	},

	fm = {
		mass        = 17,
		caliber     = 0.07,
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5,
		wind_time	= 1.8,
	},

	properties = {
		dist_min = 500,
		dist_max = 4000,
	},
}
declare_weapon(AGR_20_M282_UNGUIDED)

--#region Hydra, launchers: LAU-3/61/68/131 (incl. on BRU-33/42), M260/261, XM158

local Hydra_Mk66_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("Hydra70"),
		baseEngine		= predefined_engine("Mk66"),
	},
	variantSpecificParams = {
		["HYDRA_70_M151"] = {
			displayName			= _("Hydra 70 M151 HE"),
			displayNameShort	= "M151",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m151he",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M151") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_M151 }
		},
		["HYDRA_70_M151_M433"] = {
			displayName			= _("Hydra 70 M151 HE, M433 RC Fuze"),
			displayNameShort	= "M151",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m433",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M151") },
		},
		["HYDRA_70_M156"] = {
			displayName			= _("Hydra 70 M156 SM"),
			displayNameShort	= "M156",
			schemeName			= "nurs-marker",
			modelFileName		= "hydra_m156",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M156") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_M156 }
		},
		["HYDRA_70_M229"] = {
			displayName			= _("Hydra 70 M229 HE"),
			displayNameShort	= "M229",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m229",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M229") },
		},
		["HYDRA_70_M257"] = {
			displayName			= _("Hydra 70 M257 IL"),
			displayNameShort	= "M257",
			schemeName			= "nurs-light",
			modelFileName		= "Hydra_M257",
			payloadParams		= { launcher = { ammunition = M257_FLARE, delay = 10.1 } },
			properties = {
				dist_min	= 1000,
				dist_max	= 4000,
				max_TOF		= 10.5 -- exaggerated a bit to avoid AI overshooting the target due to non-instant deceleration of the flare element
			},
			paramOverrides	= { wsTypeOfWeaponL4 = HYDRA_70_M257 }
		},
		["HYDRA_70_M259"] = {
			displayName			= _("Hydra 70 M259 SM"),
			displayNameShort	= "M259",
			schemeName			= "nurs-marker",
			modelFileName		= "hydra_wdu4aa", -- best match available rn
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M259") },
		},
		["HYDRA_70_M274"] = {
			displayName			= _("Hydra 70 M274 TP-SM"),
			displayNameShort	= "M274",
			schemeName			= "nurs-marker",
			modelFileName		= "hydra_m156", -- best match available rn
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M274") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_M274 }
		},
		["HYDRA_70_M282"] = {
			displayName			= _("Hydra 70 M282 MPP"),
			displayNameShort	= "M282",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m282",
			payloadParams		= { warhead = predefined_warhead("HYDRA_70_M282") },
		},
		["HYDRA_70_MK1"] = {
			displayName			= _("Hydra 70 Mk 1 HE"),
			displayNameShort	= "Mk 1",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m151he",
			payloadParams		= { warhead = predefined_warhead("FFAR_Mk1") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_MK1 }
		},
		["HYDRA_70_MK5"] = {
			displayName			= _("Hydra 70 Mk 5 HEAT"),
			displayNameShort	= "Mk 5",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m151he",
			payloadParams		= { warhead = predefined_warhead("FFAR_Mk5") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_MK5 }
		},
		["HYDRA_70_MK61"] = {
			displayName			= _("Hydra 70 Mk 61 TP"),
			displayNameShort	= "Mk 61",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m156",
			payloadParams		= { warhead = predefined_warhead("FFAR_Mk61") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_MK61 }
		},
		["HYDRA_70_WTU1B"] = {
			displayName			= _("Hydra 70 WTU-1/B TP"),
			displayNameShort	= "WTU-1/B",
			schemeName			= "nurs-standard",
			modelFileName		= "hydra_m156",
			payloadParams		= { warhead = predefined_warhead("BDU") },
			paramOverrides		= { wsTypeOfWeaponL4 = HYDRA_70_WTU1B }
		},
	}
})
local MSSYS_AGR_20_M282 =
{
	category		= CAT_MISSILES,
	name			= "MSSYS_AGR_20_M282",
	user_name		= _("HYDRA-70 MPP APKWS"),
	display_name_short = "APKWS",
	scheme			= "APKWS",
	class_name		= "wAmmunitionLaserHoming",
	model			= "AGR_20",
	
	wsTypeOfWeapon 	= {wsType_Weapon,wsType_Missile,wsType_AS_Missile,WSTYPE_PLACEHOLDER},
	
	mass			= 17,
	Escort			= 1,
	Head_Type		= 4,
	sigma			= {1, 1, 1},
	H_max			= 5000.0,
	H_min			= 1.0,
	Diam			= 70.0,
	Cx_pil			= 0.00244140625,
	D_max			= 5000.0,
	D_min			= 1000.0,
	Head_Form		= 1,
	Life_Time		= 90.0,
	Nr_max			= 25,
	v_min			= 140.0,
	v_mid			= 350.0,
	Mach_max		= 3.0,
	t_b				= 0.0,
	t_acc			= 0.0,
	t_marsh			= 1.1,
	Range_max		= 5000.0,
	H_min_t			= 15.0,
	Fi_start		= 0.4,
	Fi_rak			= 3.14152,
	Fi_excort		= 1.05,
	Fi_search		= 0.7,
	OmViz_max		= 0.35,
	exhaust			= {0.78, 0.78, 0.78, 0.3};
	X_back			= -2.0,
	Y_back			= -0.0,
	Z_back			= 0.0,
	Reflection		= 0.01,
	KillDistance	= 0.0,
	M				= 15,
	
	warhead		= predefined_warhead("HYDRA_70_M282"),
	
	shape_table_data =
	{
		{
			name  = "AGR_20_M282";
			file  = "AGR_20";
			life  = 1;
			fire  = { 0, 1};
			username = _("APKWS M282"),
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},
	
	march = {
		fuel_mass   			= 3.175,
		impulse     			= 210,
		boost_time  			= 0,
		work_time   			= 1.1,
		boost_factor			= 1,
		nozzle_position 	    = {{-0.95, 0, 0}},
		nozzle_orientationXYZ   = {{0, 0, 0}},
		tail_width  			= 0.052,
		boost_tail  			= 1,
		work_tail   			= 1,
		smoke_color				= {0.5, 0.474, 0.443},
		smoke_transparency		= 0.5,
		custom_smoke_dissipation_factor		= 0.2,		
	},

	fm = {
		mass        = 17,  
		caliber     = 0.07, 
		cx_coeff    = {1, 1.5, 0.68, 0.7, 1.75},
		L           = 1.81,
		I           = 1 / 12 * 15 * 1.81 * 1.81,
		Ma          = 0.84,
		Mw          = 3.5,
		shapeName   = "AGR_20",
		wind_sigma	= 5.0,
		wind_time	= 1.1,
		dCydA		= {0.11, 0.11},
		A			= 0.36,
		Sw			= 0.04,
		maxAoa		= math.rad(8),		
		finsTau		= 0.1,
		Mx0			= 0.2,
		Ma_x		= 1.8,
		Mw_x		= 1.12,
		I_x			= 1.01,
	},
	
	seeker = {
		delay				= 1.1,
		FOV					= math.rad(70),
		max_seeker_range	= 13000,
	},

	autopilot = {
		delay				= 0.9,
		op_time				= 45,
		dV_dt				= 20,
		Knav				= 12,
		Tf					= 0.2,
		Kd					= 150,
		Kdx					= 0.2,
		Kix					= 0.8,
		gload_limit			= 10,
		fins_limit			= math.rad(20),
		fins_limit_x		= math.rad(10),
	},
}

declare_weapon(MSSYS_AGR_20_M282)

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{MSSYSMPP_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = MSSYS_AGR_20_M282.model }},
	Weight			=	17,
	wsTypeOfWeapon	=	MSSYS_AGR_20_M282.wsTypeOfWeapon,
	attribute		=	MSSYS_AGR_20_M282.wsTypeOfWeapon,
})


declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M151_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M151"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M151"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M151"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M151"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M156_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M156"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M156"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M156"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M156"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M229_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M229"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M229"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M229"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M229"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M257_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M257"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M257"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M257"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M257"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M259_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M259"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M259"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M259"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M259"].wsTypeOfWeapon,
})

declare_loadout({
	category		=	CAT_ROCKETS,
	CLSID			=	"{M274_HYDRA}",
	Count			=	1,
	Elements		=	{{ ShapeName = Hydra_Mk66_Variants["HYDRA_70_M274"].model }},
	Weight			=	Hydra_Mk66_Variants["HYDRA_70_M274"].fm.mass,
	wsTypeOfWeapon	=	Hydra_Mk66_Variants["HYDRA_70_M274"].wsTypeOfWeapon,
	attribute		=	Hydra_Mk66_Variants["HYDRA_70_M274"].wsTypeOfWeapon,
})

BatchDeclareLoadout(
	LauncherData.LAU3,
	{
		["LAU3_HE151"]	= { weaponResName = "HYDRA_70_M151",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, 186 }, },
		["LAU3_WP156"]	= { weaponResName = "HYDRA_70_M156",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, 182 }, },
		["LAU3_HE5"]	= { weaponResName = "HYDRA_70_MK5",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, 185 }, },
		["LAU3_WP61"]	= { weaponResName = "HYDRA_70_MK61",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, 184 }, },
		["LAU3_WP1B"]	= { weaponResName = "HYDRA_70_WTU1B",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, 183 } },
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	RLData.LAU61,
	{
		-- Individual LAU-61s
		["{FD90A1DC-9147-49FA-BF56-CB83EF0BD32B}"]	= { weaponResName = "HYDRA_70_M151",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_61 }, },
		["{LAU_61R}"]								= { weaponResName = "HYDRA_70_M151",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_61 }, },
		["{3DFB7321-AB0E-11d7-9897-000476191836}"]	= { weaponResName = "HYDRA_70_M156",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, Whiskey_Pete }, },
		-- LAU-61s on BRU-33
		["{BRU33_2*LAU61}"]							= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.BRU33, elemCLSID = "{FD90A1DC-9147-49FA-BF56-CB83EF0BD32B}", } },
		["{BRU33_LAU61}"]							= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.BRU33, elemCLSID = "{FD90A1DC-9147-49FA-BF56-CB83EF0BD32B}", elemCount = 1, skipTable = { 1 } } },
		-- LAU-61s on BRU-42 - triple
		["{A76344EB-32D2-4532-8FA2-0C1BDC00747E}"]	= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{FD90A1DC-9147-49FA-BF56-CB83EF0BD32B}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, MBD_3_LAU_61 }, },
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	LauncherData.LAU68,
	{
		-- Individual LAU-68s
		["{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}"]	= { weaponResName = "HYDRA_70_M151",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_M151 }, },
		["{4F977A2A-CD25-44df-90EF-164BFA2AE72F}"]	= { weaponResName = "HYDRA_70_M156",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_M156 }, },
		["{647C5F26-BDD1-41e6-A371-8DE1E4CC0E94}"]	= { weaponResName = "HYDRA_70_M257",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_M257 }, },
		["{0877B74B-5A00-4e61-BA8A-A56450BA9E27}"]	= { weaponResName = "HYDRA_70_M274",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_M274 }, },
		["{FC85D2ED-501A-48ce-9863-49D468DDD5FC}"]	= { weaponResName = "HYDRA_70_MK1",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_MK1 }, },
		["{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}"]	= { weaponResName = "HYDRA_70_MK5",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_MK5 }, },
		["{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}"]	= { weaponResName = "HYDRA_70_MK61",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_MK61 }, },
		["{1F7136CB-8120-4e77-B97B-945FF01FB67C}"]	= { weaponResName = "HYDRA_70_WTU1B",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68_HYDRA_70_WTU1B } },
		-- LAU-68s on BRU-33
		["{BRU33_2*LAU68}"]							= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.BRU33, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", } },
		["{BRU33_2*LAU68_MK5}"]						= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = RLData.BRU33, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", } },
		["{BRU33_LAU68}"]							= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.BRU33, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", elemCount = 1, skipTable = { 1 } } },
		["{BRU33_LAU68_MK5}"]						= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = RLData.BRU33, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", elemCount = 1, skipTable = { 1 } } },
		-- LAU-68s on BRU-42 - triple
		["{64329ED9-B14C-4c0b-A923-A3C911DA1527}"]	= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_M151 }, },
		["{C2593383-3CA8-4b18-B73D-0E750BCA1C85}"]	= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_M156 }, },
		["{E6966004-A525-4f47-AF94-BCFEDF8FDBDA}"]	= { weaponResName = "HYDRA_70_M257",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{647C5F26-BDD1-41e6-A371-8DE1E4CC0E94}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_M257 }, },
		["{4C044B08-886B-46c8-9B1F-AB05B3ED9C1D}"]	= { weaponResName = "HYDRA_70_M274",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{0877B74B-5A00-4e61-BA8A-A56450BA9E27}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_M274 }, },
		["{443364AE-D557-488e-9499-45EDB3BA6730}"]	= { weaponResName = "HYDRA_70_MK1",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{FC85D2ED-501A-48ce-9863-49D468DDD5FC}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_MK1 }, },
		["{9BC82B3D-FE70-4910-B2B7-3E54EFE73262}"]	= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_MK5 }, },
		["{C0FA251E-B645-4ce5-926B-F4BC20822F8B}"]	= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_MK61 }, },
		["{A1853B38-2160-4ffe-B7E9-9BF81E6C3D77}"]	= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_68x3_HYDRA_70_WTU1B }, },
		-- LAU-68s on BRU-42 - double, right
		["{BRU42LS_2*LAU68_HYDRA_70_M151_R}"]		= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_M156_R}"]		= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_M257_R}"]		= { weaponResName = "HYDRA_70_M257",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{647C5F26-BDD1-41e6-A371-8DE1E4CC0E94}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_M274_R}"]		= { weaponResName = "HYDRA_70_M274",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{0877B74B-5A00-4e61-BA8A-A56450BA9E27}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_MK1_R}"]		= { weaponResName = "HYDRA_70_MK1",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{FC85D2ED-501A-48ce-9863-49D468DDD5FC}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_MK5_R}"]		= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_MK61_R}"]		= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}", elemCount = 2 } },
		["{BRU42LS_2*LAU68_HYDRA_70_WTU1B_R}"]		= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}", elemCount = 2 } },
		-- LAU-68s on BRU-42 - double, left
		["{BRU42LS_2*LAU68_HYDRA_70_M151_L}"]		= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_M156_L}"]		= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_M257_L}"]		= { weaponResName = "HYDRA_70_M257",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{647C5F26-BDD1-41e6-A371-8DE1E4CC0E94}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_M274_L}"]		= { weaponResName = "HYDRA_70_M274",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{0877B74B-5A00-4e61-BA8A-A56450BA9E27}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_MK1_L}"]		= { weaponResName = "HYDRA_70_MK1",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{FC85D2ED-501A-48ce-9863-49D468DDD5FC}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_MK5_L}"]		= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_MK61_L}"]		= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU68_HYDRA_70_WTU1B_L}"]		= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}", elemCount = 2, skipTable = { 2 } } },
		-- LAU-68s on TER-9A - double, right
		["{TER_9A_2R*LAU-68_M151}"]					= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-68_M156}"]					= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-68_MK5}"]					= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = RLData.TER9, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-68_MK61}"]					= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-68_WTU-1/B}"]				= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}", elemCount = 2, skipTable = { 2 } } },
		-- LAU-68s on TER-9A - double, left
		["{TER_9A_2L*LAU-68_M151}"]					= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{A021F29D-18AB-4d3e-985C-FC9C60E35E9E}", elemCount = 2 } },
		["{TER_9A_2L*LAU-68_M156}"]					= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{4F977A2A-CD25-44df-90EF-164BFA2AE72F}", elemCount = 2 } },
		["{TER_9A_2L*LAU-68_MK5}"]					= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = RLData.TER9, elemCLSID = "{174C6E6D-0C3D-42ff-BCB3-0853CB371F5C}", elemCount = 2 } },
		["{TER_9A_2L*LAU-68_MK61}"]					= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{65396399-9F5C-4ec3-A7D2-5A8F4C1D90C4}", elemCount = 2 } },
		["{TER_9A_2L*LAU-68_WTU-1/B}"]				= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{1F7136CB-8120-4e77-B97B-945FF01FB67C}", elemCount = 2 } },
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	LauncherData.LAU131,
	{
		-- Individual LAU-131s
		["{69926055-0DA8-4530-9F2F-C86B157EA9F6}"]	= { weaponResName = "HYDRA_70_M151",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_M151 }, },
		["{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}"]	= { weaponResName = "HYDRA_70_M156",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_M156 }, },
		["{DAD45FE5-CFF0-4a2b-99D4-5D044D3BC22F}"]	= { weaponResName = "HYDRA_70_M257",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_M257 }, },
		["{6D6D5C07-2A90-4a68-9A74-C5D0CFFB05D9}"]	= { weaponResName = "HYDRA_70_M274",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_M274 }, },
		["{D22C2D63-E5C9-4247-94FB-5E8F3DE22B71}"]	= { weaponResName = "HYDRA_70_MK1",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_MK1 }, },
		["{319293F2-392C-4617-8315-7C88C22AF7C4}"]	= { weaponResName = "HYDRA_70_MK5",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_MK5 }, },
		["{1CA5E00B-D545-4ff9-9B53-5970E292F14D}"]	= { weaponResName = "HYDRA_70_MK61",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_MK61 }, },
		["{DDCE7D70-5313-4181-8977-F11018681662}"]	= { weaponResName = "HYDRA_70_WTU1B",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131_HYDRA_70_WTU1B }, },
		-- LAU-131s on BRU-42 - triple
		["LAU_131x3_HYDRA_70_M151"]					= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_M151 }, },
		["LAU_131x3_HYDRA_70_M156"]					= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_M156 }, },
		["LAU_131x3_HYDRA_70_M257"]					= { weaponResName = "HYDRA_70_M257",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{DAD45FE5-CFF0-4a2b-99D4-5D044D3BC22F}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_M257 }, },
		["LAU_131x3_HYDRA_70_M274"]					= { weaponResName = "HYDRA_70_M274",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{6D6D5C07-2A90-4a68-9A74-C5D0CFFB05D9}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_M274 }, },
		["LAU_131x3_HYDRA_70_MK1"]					= { weaponResName = "HYDRA_70_MK1",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{D22C2D63-E5C9-4247-94FB-5E8F3DE22B71}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_MK1 }, },
		["LAU_131x3_HYDRA_70_MK5"]					= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_MK5 }, },
		["LAU_131x3_HYDRA_70_MK61"]					= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_MK61 }, },
		["LAU_131x3_HYDRA_70_WTU1B"]				= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{DDCE7D70-5313-4181-8977-F11018681662}" }, attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131x3_HYDRA_70_WTU1B }, },
		-- LAU-131s on BRU-42 - double, right
		["{BRU42LS_2*LAU131_HYDRA_70_M151_R}"]		= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_M156_R}"]		= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_M257_R}"]		= { weaponResName = "HYDRA_70_M257",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{DAD45FE5-CFF0-4a2b-99D4-5D044D3BC22F}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_M274_R}"]		= { weaponResName = "HYDRA_70_M274",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{6D6D5C07-2A90-4a68-9A74-C5D0CFFB05D9}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_MK1_R}"]		= { weaponResName = "HYDRA_70_MK1",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{D22C2D63-E5C9-4247-94FB-5E8F3DE22B71}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_MK5_R}"]		= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_MK61_R}"]		= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}", elemCount = 2 } },
		["{BRU42LS_2*LAU131_HYDRA_70_WTU1B_R}"]		= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{DDCE7D70-5313-4181-8977-F11018681662}", elemCount = 2 } },
		-- LAU-131s on BRU-42 - double, left
		["{BRU42LS_2*LAU131_HYDRA_70_M151_L}"]		= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_M156_L}"]		= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_M257_L}"]		= { weaponResName = "HYDRA_70_M257",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{DAD45FE5-CFF0-4a2b-99D4-5D044D3BC22F}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_M274_L}"]		= { weaponResName = "HYDRA_70_M274",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{6D6D5C07-2A90-4a68-9A74-C5D0CFFB05D9}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_MK1_L}"]		= { weaponResName = "HYDRA_70_MK1",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{D22C2D63-E5C9-4247-94FB-5E8F3DE22B71}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_MK5_L}"]		= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_MK61_L}"]		= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}", elemCount = 2, skipTable = { 2 } } },
		["{BRU42LS_2*LAU131_HYDRA_70_WTU1B_L}"]		= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = LauncherData.BRU42, elemCLSID = "{DDCE7D70-5313-4181-8977-F11018681662}", elemCount = 2, skipTable = { 2 } } },
		-- LAU-131s on BRU-42 - double, right
		["{TER_9A_2R*LAU-131_M151}"]				= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-131_M156}"]				= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-131_MK5}"]					= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = RLData.TER9, elemCLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-131_MK61}"]				= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}", elemCount = 2, skipTable = { 2 } } },
		["{TER_9A_2R*LAU-131_WTU-1/B}"]				= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{DDCE7D70-5313-4181-8977-F11018681662}", elemCount = 2, skipTable = { 2 } } },
		-- LAU-131s on BRU-42 - double, left
		["{TER_9A_2L*LAU-131_M151}"]				= { weaponResName = "HYDRA_70_M151",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{69926055-0DA8-4530-9F2F-C86B157EA9F6}", elemCount = 2 } },
		["{TER_9A_2L*LAU-131_M156}"]				= { weaponResName = "HYDRA_70_M156",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{2AF2EC3F-9065-4de5-93E1-1739C9A71EF7}", elemCount = 2 } },
		["{TER_9A_2L*LAU-131_MK5}"]					= { weaponResName = "HYDRA_70_MK5",		multiRackParams = { MER = RLData.TER9, elemCLSID = "{319293F2-392C-4617-8315-7C88C22AF7C4}", elemCount = 2 } },
		["{TER_9A_2L*LAU-131_MK61}"]				= { weaponResName = "HYDRA_70_MK61",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{1CA5E00B-D545-4ff9-9B53-5970E292F14D}", elemCount = 2 } },
		["{TER_9A_2L*LAU-131_WTU-1/B}"]				= { weaponResName = "HYDRA_70_WTU1B",	multiRackParams = { MER = RLData.TER9, elemCLSID = "{DDCE7D70-5313-4181-8977-F11018681662}", elemCount = 2 } },
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	LauncherData.M260,
	{
		["{M260_M151}"]				= { weaponResName = "HYDRA_70_M151" },
		["{M260_M156}"]				= { weaponResName = "HYDRA_70_M156" },
		["{M260_M229}"]				= { weaponResName = "HYDRA_70_M229" },
		["{M260_M257}"]				= { weaponResName = "HYDRA_70_M257" },
		["{M260_M259}"]				= { weaponResName = "HYDRA_70_M259" },
		["{M260_M274}"]				= { weaponResName = "HYDRA_70_M274" },
		["{M260_A_M151_B_M156}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M156"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M156_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M156",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M151_B_M257}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M257_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M257",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M151_B_M259}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M259"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M259_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M259",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M151_B_M274}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 4,
				["HYDRA_70_M274"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M151_HYDRA}", ["B"] = "{M274_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M151; B - M274",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M260_A_M229_B_M156}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M229"] = 4,
				["HYDRA_70_M156"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M229_HYDRA}", ["B"] = "{M156_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M229; B - M156",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M229"
		},
		["{M260_A_M229_B_M257}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M229"] = 4,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M229_HYDRA}", ["B"] = "{M257_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M229; B - M257",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M229"
		},
		["{M260_A_M229_B_M259}"]	= {
			multiWeaponTable	= {
				["HYDRA_70_M229"] = 4,
				["HYDRA_70_M259"] = 3,
			},
			Elements			= fillZonedLauncher(LauncherData.M260, { ["A"] = "{M229_HYDRA}", ["B"] = "{M259_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A - M229; B - M259",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M229"
		},
		["M260_HYDRA"]				= { weaponResName = "HYDRA_70_MK5",		Elements = ContainerWithSubshapes("M260", "hydra_m151he", 7, "tube_%d"),	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131 } },
		["M260_HYDRA_WP"]			= { weaponResName = "HYDRA_70_M156",	Elements = ContainerWithSubshapes("M260", "hydra_m156", 7, "tube_%d"),		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_131WP } },
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	RLData.M261,
	{
		["M261_MK151"]			= { weaponResName = "HYDRA_70_M151" },
		["{M261_M151_M433}"]	= { weaponResName = "HYDRA_70_M151_M433" },
		["M261_MK156"]			= { weaponResName = "HYDRA_70_M156" },
		["{M261_M229}"]			= { weaponResName = "HYDRA_70_M229" },
		["{M261_M257}"]			= { weaponResName = "HYDRA_70_M257" },
		["{M261_M259}"]			= { weaponResName = "HYDRA_70_M259" },
		["{M261_M274}"]			= { weaponResName = "HYDRA_70_M274" },
		["{M261_M282}"]			= { weaponResName = "HYDRA_70_M282" },
		["{M261_OUTBOARD_AB_M151_E_M274}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M274"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["A"] = "{M151_HYDRA}", ["B"] = "{M151_HYDRA}", ["E"] = "{M274_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A/B - M151; E - M274",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M261_OUTBOARD_AB_M151_E_M257}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["A"] = "{M151_HYDRA}", ["B"] = "{M151_HYDRA}", ["E"] = "{M257_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": A/B - M151; E - M257",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M261_INBOARD_DE_M151_C_M274}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M274"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["C"] = "{M274_HYDRA}", ["D"] = "{M151_HYDRA}", ["E"] = "{M151_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": C - M274; D/E - M151",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
		["{M261_INBOARD_DE_M151_C_M257}"] = {
			multiWeaponTable	= {
				["HYDRA_70_M151"] = 16,
				["HYDRA_70_M257"] = 3,
			},
			Elements			= fillZonedLauncher(RLData.M261, { ["C"] = "{MSSYSMPP_HYDRA}", ["D"] = "{M151_HYDRA}", ["E"] = "{M151_HYDRA}" }),
			displayNamePstfx	= ", " .. LocStr.PodZones .. ": C - MPP APKWS; D/E - M151",
			wsTypeOfWeapon		= "weapons.nurs.HYDRA_70_M151"
		},
	},
	Hydra_Mk66_Variants
)

BatchDeclareLoadout(
	RLData.XM158,
	{
		["XM158_M151"]	= { weaponResName = "HYDRA_70_M151",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, XM_158_HYDRA_70_M151 }, },
		["XM158_M156"]	= { weaponResName = "HYDRA_70_M156",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, XM_158_HYDRA_70_M156 }, },
		["XM158_M257"]	= { weaponResName = "HYDRA_70_M257",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, XM_158_HYDRA_70_M257 }, },
		["XM158_M274"]	= { weaponResName = "HYDRA_70_M274",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, XM_158_HYDRA_70_M274 }, },
		["XM158_MK1"]	= { weaponResName = "HYDRA_70_MK1",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, XM_158_HYDRA_70_MK1 }, },
		["XM158_MK5"]	= { weaponResName = "HYDRA_70_MK5",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, XM_158_HYDRA_70_MK5 }, },
	},
	Hydra_Mk66_Variants
)

--#endregion

--#region Zuni, launchers: LAU-10 (incl. on BRU-33)

local Zuni_Mk71_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("Zuni_WAFAR"),
		baseEngine	= predefined_engine("Mk71Mod1"),
		properties	= { dist_min = 1200, dist_max = 4000 }
	},
	variantSpecificParams = {
		["Zuni_127"] = {
			displayName			= _("Zuni Mk. 24 Mod. 1 HE"),
			displayNameShort	= "Zuni",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("Zuni_Mk24Mod1") },
			modelFileName		= "zuni",
			paramOverrides		= { wsTypeOfWeaponL4 = Zuni_127 }
		},
	}
})

BatchDeclareLoadout(
	RLData.LAU10,
	{
		-- Individual LAU-10s
		["{F3EFE0AB-E91A-42D8-9CA2-B63C91ED570A}"]	= { weaponResName = "Zuni_127",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_10 }, },
		["{LAU_10R}"]								= { weaponResName = "Zuni_127",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, LAU_10 }, },
		-- LAU-10s on BRU-33
		["{BRU33_2*LAU10}"]							= { weaponResName = "Zuni_127",	multiRackParams = { MER = RLData.BRU33, elemCLSID = "{F3EFE0AB-E91A-42D8-9CA2-B63C91ED570A}", } },
		["{BRU33_LAU10}"]							= { weaponResName = "Zuni_127",	multiRackParams = { MER = RLData.BRU33, elemCLSID = "{F3EFE0AB-E91A-42D8-9CA2-B63C91ED570A}", elemCount = 1, skipTable = { 1 } } },
	},
	Zuni_Mk71_Variants
)

--#endregion

--#region ARF-8, launchers: LR-25

local ARF8M3_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("ARF8M3"),
		baseEngine	= predefined_engine("ARF8M3"),
		properties	= { dist_min = 450, dist_max = 4500 }
	},
	variantSpecificParams = {
		["ARF8M3API"] = {
			displayName			= _("ARF-8/M3 API"),
			displayNameShort	= "ARF-8",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("ARF8M3API")},
			modelFileName		= "ARF8M3_API",
		},
		["ARF8M3HEI"] = {
			displayName			= _("ARF-8/M3 HEI Heavy"),
			displayNameShort	= "ARF-8",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("ARF8M3HEIHeavy")},
			modelFileName		= "ARF8M3_HEI",
		},
		["ARF8M3TPSM"] = {
			displayName			= _("ARF-8/M3 TP-SM"),
			displayNameShort	= "ARF-8",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("ARF8M3TPSM")},
			modelFileName		= "ARF8M3_TP",
		},
	}
})

BatchDeclareLoadout(
	RLData.LR25,
	{
		["{LR25_ARF8M3_API}"]	= { weaponResName = "ARF8M3API", },
		["{LR25_ARF8M3_HEI}"]	= { weaponResName = "ARF8M3HEI", },
		["{LR25_ARF8M3_TPSM}"]	= { weaponResName = "ARF8M3TPSM", },
	},
	ARF8M3_Variants
)

--#endregion

--#region SNEB-68, launchers: MATRA F1/F4, Telson 8

local SNEB68_F1B_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("SNEB68"),
		baseEngine		= predefined_engine("SNEBType25F1B"),
		modelFileName	= "SA342_SNEB68"
	},
	variantSpecificParams = {
		["SNEB_TYPE250_F1B"] = {
			displayName			= _("SNEB Type 250 F1B TP-SM"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType20") },
		},
		["SNEB_TYPE251_F1B"] = {
			displayName			= _("SNEB Type 251 F1B HE"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType21") },
		},
		["SNEB_TYPE252_F1B"] = {
			displayName			= _("SNEB Type 252 F1B TP"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType22") },
		},
		["SNEB_TYPE253_F1B"] = {
			displayName			= _("SNEB Type 253 F1B HEAT"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType23") },
		},
		["SNEB_TYPE254_F1B_RED"] = {
			displayName			= _("SNEB Type 254 F1B SM Red"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType24Red") },
		},
		["SNEB_TYPE254_F1B_YELLOW"] = {
			displayName			= _("SNEB Type 254 F1B SM Yellow"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType24Yellow") },
		},
		["SNEB_TYPE254_F1B_GREEN"] = {
			displayName			= _("SNEB Type 254 F1B SM Green"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType24Green") },
		},
		["SNEB_TYPE256_F1B"] = {
			displayName			= _("SNEB Type 256 F1B HE/Frag"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType26") },
		},
		["SNEB_TYPE257_F1B"] = {
			displayName			= _("SNEB Type 257 F1B HE/Frag Lg Whd"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType27") },
		},
		["SNEB_TYPE259E_F1B"] = {
			displayName			= _("SNEB Type 259E F1B IL"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-light",
			payloadParams		= { launcher = { ammunition = S_8OM_FLARE, delay = 10.1 } },
		},
	}
})

local SNEB68_Rotary_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM			= predefined_fm("SNEB68"),
		baseEngine		= predefined_engine("SNEBType25H1"),
		modelFileName	= "SA342_SNEB68",
	},
	variantSpecificParams = {
		["SNEB_TYPE251_H1"] = {
			displayName			= _("SNEB Type 251 H1 HE"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType21") },
		},
		["SNEB_TYPE252_H1"] = {
			displayName			= _("SNEB Type 252 H1 TP"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType22") },
		},
		["SNEB_TYPE253_H1"] = {
			displayName			= _("SNEB Type 253 H1 HEAT"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType23") },
		},
		["SNEB_TYPE254_H1_RED"] = {
			displayName			= _("SNEB Type 254 H1 SM Red"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType24Red") },
		},
		["SNEB_TYPE254_H1_YELLOW"] = {
			displayName			= _("SNEB Type 254 H1 SM Yellow"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType24Yellow") },
		},
		["SNEB_TYPE254_H1_GREEN"] = {
			displayName			= _("SNEB Type 254 H1 SM Green"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-marker",
			payloadParams		= { warhead = predefined_warhead("SNEBType24Green") },
		},
		["SNEB_TYPE256_H1"] = {
			displayName			= _("SNEB Type 256 H1 HE/Frag"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType26") },
		},
		["SNEB_TYPE257_H1"] = {
			displayName			= _("SNEB Type 257 H1 HE/Frag Lg Whd"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("SNEBType27") },
		},
		["SNEB_TYPE259E_H1"] = {
			displayName			= _("SNEB Type 259E H1 IL"),
			displayNameShort	= "SNEB-68",
			schemeName			= "nurs-light",
			payloadParams		= { launcher = { ammunition = S_8OM_FLARE, delay = 10.1 } },
		},
	}
})

BatchDeclareLoadout(
	RLData.MATRAF1,
	{
		["{MATRA_F1_SNEBT250}"]			= { weaponResName = "SNEB_TYPE250_F1B" },
		["{MATRA_F1_SNEBT251}"]			= { weaponResName = "SNEB_TYPE251_F1B" },
		["{MATRA_F1_SNEBT252}"]			= { weaponResName = "SNEB_TYPE252_F1B" },
		["{MATRA_F1_SNEBT253}"]			= { weaponResName = "SNEB_TYPE253_F1B" },
		["{MATRA_F1_SNEBT254_RED}"]		= { weaponResName = "SNEB_TYPE254_F1B_RED" },
		["{MATRA_F1_SNEBT254_YELLOW}"]	= { weaponResName = "SNEB_TYPE254_F1B_YELLOW" },
		["{MATRA_F1_SNEBT254_GREEN}"]	= { weaponResName = "SNEB_TYPE254_F1B_GREEN" },
		["{MATRA_F1_SNEBT256}"]			= { weaponResName = "SNEB_TYPE256_F1B" },
		["{MATRA_F1_SNEBT257}"]			= { weaponResName = "SNEB_TYPE257_F1B" },
		["{MATRA_F1_SNEBT259E}"]		= { weaponResName = "SNEB_TYPE259E_F1B" },
	},
	SNEB68_F1B_Variants
)

BatchDeclareLoadout(
	RLData.MATRAF4,
	{
		["{MATRA_F4_SNEBT250}"]			= { weaponResName = "SNEB_TYPE250_F1B" },
		["{MATRA_F4_SNEBT251}"]			= { weaponResName = "SNEB_TYPE251_F1B" },
		["{MATRA_F4_SNEBT252}"]			= { weaponResName = "SNEB_TYPE252_F1B" },
		["{MATRA_F4_SNEBT253}"]			= { weaponResName = "SNEB_TYPE253_F1B" },
		["{MATRA_F4_SNEBT254_RED}"]		= { weaponResName = "SNEB_TYPE254_F1B_RED" },
		["{MATRA_F4_SNEBT254_YELLOW}"]	= { weaponResName = "SNEB_TYPE254_F1B_YELLOW" },
		["{MATRA_F4_SNEBT254_GREEN}"]	= { weaponResName = "SNEB_TYPE254_F1B_GREEN" },
		["{MATRA_F4_SNEBT256}"]			= { weaponResName = "SNEB_TYPE256_F1B" },
		["{MATRA_F4_SNEBT257}"]			= { weaponResName = "SNEB_TYPE257_F1B" },
		["{MATRA_F4_SNEBT259E}"]		= { weaponResName = "SNEB_TYPE259E_F1B" },
	},
	SNEB68_F1B_Variants
)

-- Type 250 is available only with F1B motor
SNEB68_Rotary_Variants["SNEB_TYPE250_F1B"] = SNEB68_F1B_Variants["SNEB_TYPE250_F1B"]

BatchDeclareLoadout(
	RLData.TELSON8,
	{
		["{TELSON8_SNEBT250}"]			= { weaponResName = "SNEB_TYPE250_F1B", Picture = "SNEB68_Smoke.png" },
		["{TELSON8_SNEBT251}"]			= { weaponResName = "SNEB_TYPE251_H1" },
		["{TELSON8_SNEBT252}"]			= { weaponResName = "SNEB_TYPE252_H1" },
		["{TELSON8_SNEBT253}"]			= { weaponResName = "SNEB_TYPE253_H1" },
		["{TELSON8_SNEBT254_RED}"]		= { weaponResName = "SNEB_TYPE254_H1_RED", Picture = "SNEB68_Smoke.png", PictureBlendColor = "0xff0000ff" },
		["{TELSON8_SNEBT254_YELLOW}"]	= { weaponResName = "SNEB_TYPE254_H1_YELLOW", Picture = "SNEB68_Smoke.png", PictureBlendColor = "0xffff00ff" },
		["{TELSON8_SNEBT254_GREEN}"]	= { weaponResName = "SNEB_TYPE254_H1_GREEN", Picture = "SNEB68_Smoke.png", PictureBlendColor = "0x00ff00ff" },
		["{TELSON8_SNEBT256}"]			= { weaponResName = "SNEB_TYPE256_H1" },
		["{TELSON8_SNEBT257}"]			= { weaponResName = "SNEB_TYPE257_H1" },
		["{TELSON8_SNEBT259E}"]			= { weaponResName = "SNEB_TYPE259E_H1" },
		-- Backwards-compatibility section
		["{LAU_SNEB68_WP}"]				= { weaponResName = "SNEB_TYPE250_F1B", Picture = "SNEB68_Smoke.png" },
		["{LAU_SNEB68G}"]				= { weaponResName = "SNEB_TYPE253_H1" },
	},
	SNEB68_Rotary_Variants
)

--#endregion

--#region S-5, launchers: UB-16, UB-32A

local S5_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("S5"),
		baseEngine	= predefined_engine("S5"),
		properties	= { dist_min = 600, dist_max = 2500 },
	},
	variantSpecificParams = {
		["C_5"] = {
			displayName			= _("S-5KO HEAT/Frag"),
			displayNameShort	= LocStr.dNS.S5,
			modelFileName		= "s-5ko",
			paramOverrides		= { wsTypeOfWeaponL4 = C_5 },
			payloadParams		= { warhead = predefined_warhead("S_5KO") },
			schemeName			= "nurs-standard",
		},
		["S_5KP"] = {
			displayName			= _("S-5KP HEAT/Frag"),
			displayNameShort	= LocStr.dNS.S5,
			modelFileName		= "c-5",
			payloadParams		= { warhead = predefined_warhead("S_5KP") },
			schemeName			= "nurs-standard",
		},
		["S_5M"] = {
			displayName			= _("S-5M HE"),
			displayNameShort	= LocStr.dNS.S5,
			modelFileName		= "s-5m",
			payloadParams		= { warhead = predefined_warhead("S_5M") },
			schemeName			= "nurs-standard",
		},
	}
})

BatchDeclareLoadout(
	RLData.UB16UMP,
	{
		["{UB-16-57UMP}"]	= { weaponResName = "C_5" },
		["{UB16_S5KP}"]		= { weaponResName = "S_5KP",	Picture = "rus_UB-16_S-5K.png" },
		["{UB16_S5M}"]		= { weaponResName = "S_5M",		Picture = "rus_UB-16_S-5M.png" },
	},
	S5_Variants
)

BatchDeclareLoadout(
	RLData.UB32A,
	{
		["{637334E4-AB5A-47C0-83A6-51B7F1DF3CD5}"]	= { weaponResName = "C_5",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, UB_32_1 } },
		["{UB32A_S5KP}"]							= { weaponResName = "S_5KP",	Picture = "rus_UB-32_S-5K.png" },
		["{UB32A_S5M}"]								= { weaponResName = "S_5M",		Picture = "rus_UB-32_S-5M.png" },
	},
	S5_Variants
)

BatchDeclareLoadout(
	RLData.UB32A24,
	{
		["{UB-32A-24}"]		= { weaponResName = "C_5" },
		["{UB32A24_S5KP}"]	= { weaponResName = "S_5KP",	Picture = "rus_UB-32_S-5K.png" },
		["{UB32A24_S5M}"]	= { weaponResName = "S_5M",		Picture = "rus_UB-32_S-5M.png" },
	},
	S5_Variants
)

--#endregion

--#region S-8, launchers: B-8V20A, B-8M1 (incl. twin-pylon versions)

local S8_FFAR_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("S8"),
		baseEngine	= predefined_engine("9GSch4421"),
		properties	= { dist_min = 600, dist_max = 3000 }
	},
	variantSpecificParams = {
		["C_8"] = {
			displayName			= _("S-8KOM HEAT/Frag"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-standard",
			modelFileName		= "s-8_kom",
			payloadParams		= { warhead = predefined_warhead("S_8KOM") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8 }
		},
		["C_8OM"] = {
			displayName			= _("S-8OM IL"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-light",
			modelFileName		= "s-8_om",
			payloadParams		= { launcher = { ammunition = S_8OM_FLARE, delay = 15 } },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8OM },
			properties			= {
				dist_min	= 4000,
				dist_max	= 4500,
				max_TOF		= 15.1 -- exaggerated a bit to avoid AI overshooting the target due to non-instant deceleration of the flare element
			}
		},
		["C_8CM"] = {
			displayName			= _("S-8TsM SM Orange"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_ORANGE") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM }
		},
		["C_8CM_GN"] = {
			displayName			= _("S-8TsM SM Green"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_GREEN") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM_GN }
		},
		["C_8CM_RD"] = {
			displayName			= _("S-8TsM SM Red"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_RED") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM_RD }
		},
		["C_8CM_WH"] = {
			displayName			= _("S-8TsM SM White"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_WHITE") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM_WH }
		},
		["C_8CM_BU"] = {
			displayName			= _("S-8TsM SM Blue"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_BLUE") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM_BU }
		},
		["C_8CM_YE"] = {
			displayName			= _("S-8TsM SM Yellow"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_YELLOW") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM_YE }
		},
		["C_8CM_VT"] = {
			displayName			= _("S-8TsM SM Violet"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-marker",
			modelFileName		= "s-8_sm",
			payloadParams		= { warhead = predefined_warhead("S_8TsM_VIOLET") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8CM_VT }
		},
	}
})

local S8_WAFAR_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("S8"),
		baseEngine	= predefined_engine("S8_WAFAR"),
		properties	= { dist_min = 600, dist_max = 4000 }
	},
	variantSpecificParams = {
		["C_8OFP2"] = {
			displayName			= _("S-8OFP2 MPP"),
			displayNameShort	= LocStr.dNS.S8,
			schemeName			= "nurs-standard",
			modelFileName		= "s-8_ofp2",
			payloadParams		= { warhead = predefined_warhead("S_8OFP") },
			paramOverrides		= { wsTypeOfWeaponL4 = C_8OFP2 }
		},
	}
})

local All_S8_Variants = S8_FFAR_Variants

for descName, desc in pairs(S8_WAFAR_Variants) do
	All_S8_Variants[descName] = desc
end

BatchDeclareLoadout(
	RLData.B8V20A,
	{
		["{6A4B9E69-64FE-439a-9163-3A87FB6A4D81}"]	= { weaponResName = "C_8",		Picture = "rus_B-8V20A_S-8KOM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A }, },
		["B_8V20A_CM"]								= { weaponResName = "C_8CM",	Picture = "rus_B-8V20A_S-8TsM_OG.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_GN"]							= { weaponResName = "C_8CM_GN",	Picture = "rus_B-8V20A_S-8TsM_GN.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_RD"]							= { weaponResName = "C_8CM_RD",	Picture = "rus_B-8V20A_S-8TsM_RD.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_WH"]							= { weaponResName = "C_8CM_WH",	Picture = "rus_B-8V20A_S-8TsM_WH.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_BU"]							= { weaponResName = "C_8CM_BU",	Picture = "rus_B-8V20A_S-8TsM_BU.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_YE"]							= { weaponResName = "C_8CM_YE",	Picture = "rus_B-8V20A_S-8TsM_YE.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_CM_VT"]							= { weaponResName = "C_8CM_VT",	Picture = "rus_B-8V20A_S-8TsM_VT.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_WP }, },
		["B_8V20A_OFP2"]							= { weaponResName = "C_8OFP2",	Picture = "rus_B-8V20A_S-8OFP2.png",	attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_OFP2 }, },
		["B_8V20A_OM"]								= { weaponResName = "C_8OM",	Picture = "rus_B-8V20A_S-8OM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8V20A_OM }, },
	},
	All_S8_Variants
)

BatchDeclareLoadout(
	RLData.B8M1,
	{
		["{F72F47E5-C83A-4B85-96ED-D3E46671EE9A}"]	= { weaponResName = "C_8",		Picture = "rus_B-8M1_S-8KOM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_20 }, },
		["{3DFB7320-AB0E-11d7-9897-000476191836}"]	= { weaponResName = "C_8CM",	Picture = "rus_B-8M1_S-8TsM.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_20CM }, },
		["B-8M1 - 20 S-8OFP2"]						= { weaponResName = "C_8OFP2",	Picture = "rus_B-8M1_S-8OFP2.png",		attribute = { wsType_Weapon, wsType_NURS, wsType_Container, B_8M1_OFP2 }, },
		["{B8M1_S8OM}"]								= { weaponResName = "C_8OM",	Picture = "rus_B-8M1_S-8OM.png", },
		["{TWIN_B_8M1_S_8KOM}"]						= { weaponResName = "C_8",		Picture = "rus_B-8M1_S-8KOM.png",		multiRackParams = { MER = RLData.Su27TwinPylon, elemCLSID = "{F72F47E5-C83A-4B85-96ED-D3E46671EE9A}" }},
		["{TWIN_B_8M1_S_8TsM}"]						= { weaponResName = "C_8CM",	Picture = "rus_B-8M1_S-8TsM.png",		multiRackParams = { MER = RLData.Su27TwinPylon, elemCLSID = "{3DFB7320-AB0E-11d7-9897-000476191836}" }},
		["{TWIN_B_8M1_S_8_OFP2}"]					= { weaponResName = "C_8OFP2",	Picture = "rus_B-8M1_S-8OFP2.png",		multiRackParams = { MER = RLData.Su27TwinPylon, elemCLSID = "B-8M1 - 20 S-8OFP2" }},
	},
	All_S8_Variants
)

--#endregion

--#region S-13, launchers: B-13L (incl. twin-pylon versions)

local S13_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("S13"),
		baseEngine	= predefined_engine("S13_FFAR"),
		properties	= { dist_min = 1100, dist_max = 4000 },
	},
	variantSpecificParams = {
		["C_13"] = {
			displayName			= _("S-13OF Blast/Fragmentation"),
			displayNameShort	= LocStr.dNS.S13,
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("C_13") },
			modelFileName		= "c-13",
			paramOverrides		= { wsTypeOfWeaponL4 = C_13 }
		},
	}
})

BatchDeclareLoadout(
	RLData.B13L,
	{
		["{FC56DF80-9B09-44C5-8976-DCFAFF219062}"] = {
			attribute		= { wsType_Weapon, wsType_NURS, wsType_Container, UB_13 },
			weaponResName	= "C_13",
			
		},
		["{TWIN_B13L_5OF}"] = {
			multiRackParams	= { MER = RLData.Su27TwinPylon, elemCLSID = "{FC56DF80-9B09-44C5-8976-DCFAFF219062}" },
			weaponResName	= "C_13"
		},
	},
	S13_Variants
)

--#endregion

--#region S-24, launchers: APU-68

local S_24 = {
	category			= CAT_ROCKETS,
	name				= "C_24",
	user_name			= _("S-24B"),
	display_name_short	= _("S-24"),
	scheme				= "nurs-standard",
	class_name			= "wAmmunitionNURS",
	model				= "c-24",
	wsTypeOfWeapon		= {wsType_Weapon, wsType_NURS, wsType_Rocket, C_24},

	fm = {
		mass					= 235,
		caliber					= 0.24,
		cx_coeff				= {1, 1.889, 0.67, 0.33, 1.8},
		L						= 2.22,
		I						= 96.5145,
		Ma						= 0.1720553,
		Mw						= 6.106276,
		shapeName				= "",
		rail_open				= true,
		wind_time				= 1.3,
		wind_sigma				= 15,
		wind_sigma_z_coeff		= 1,
	},

	engine = {
		fuel_mass				= 51.8,
		impulse					= 170,
		boost_time				= 0,
		work_time				= 1.3,
		boost_factor			= 1,
		nozzle_position			= {{-1.14, 0.039, -0.068}, {-1.14, 0.078, 0}, {-1.14, 0.039, 0.068}, {-1.14, -0.039, -0.068}, {-1.14, -0.078, 0}, {-1.14, -0.039, 0.068}},
		nozzle_orientationXYZ	= {{0, 0, 0}},
		tail_width				= 0.074,
		boost_tail				= 1,
		work_tail				= 1,

		smoke_color				= {0.5, 0.5, 0.5},
		smoke_transparency		= 0.5,
	},

	warhead = predefined_warhead("C_24"),

	properties = {
		dist_min = 2000,
		dist_max = 3000
	},

	shape_table_data = {{
		name		= "S-24";
		file		= "C-24";
		username	= _("S-24B"),
		index		= C_24,
	}},
}
declare_weapon(S_24)

declare_loadout({
	category		= CAT_ROCKETS,
	CLSID			= "{1FA14DEA-8CDB-45AD-88A8-EC068DF1E65A}",
	wsTypeOfWeapon	= S_24.wsTypeOfWeapon,
	attribute		= S_24.wsTypeOfWeapon,
	Count			= 1,
	Cx_pil			= 0.0005,
	Picture			= "RBK250.png",
	displayName		= LocStr.S24B,
	Weight			= 235,
	Weight_Empty	= 0,
	Elements		= {{ ShapeName = S_24.model, Position = { 0, -0.14, 0 } }}
})

declare_loadout({
	category		= CAT_ROCKETS,
	CLSID			= "{3858707D-F5D5-4bbb-BDD8-ABB0530EBC7C}",
	wsTypeOfWeapon	= S_24.wsTypeOfWeapon,
	attribute		= { wsType_Weapon, wsType_NURS, wsType_Container, APU_68_C_24 },
	Count			= 1,
	Cx_pil			= 0.001,
	Picture			= "RBK250.png",
	displayName		= ("%s - %s"):format(LocStr.APU68, LocStr.S24B),
	Weight			= 60 + 235,
	Weight_Empty	= 60,
	Elements		= {{ ShapeName = "APU-68", IsAdapter = true }, { ShapeName = S_24.model, connector_name = "Point01" }}
})

declare_loadout({
	category		= CAT_ROCKETS,
	CLSID			= "{APU_68_S-24}",
	wsTypeOfWeapon	= S_24.wsTypeOfWeapon,
	attribute		= { wsType_Weapon, wsType_NURS, wsType_Container, WSTYPE_PLACEHOLDER },
	Count			= 1,
	Cx_pil			= 0.0002,
	Picture			= "RBK250.png",
	displayName		= ("%s - %s"):format(LocStr.APU68UM3, LocStr.S24B),
	Weight			= 38.5 + 235,
	Weight_Empty	= 38.5,
	Elements		= {{ ShapeName = "apu-68um3",IsAdapter = true }, { payload_CLSID = "{1FA14DEA-8CDB-45AD-88A8-EC068DF1E65A}", connector_name = "Point Pilon s24" }}
})

--#endregion

--#region S-25, launchers: O-25 (incl. twin-pylon versions)

local S25_Variants = BatchDeclareRockets({
	commonParams = {
		baseFM		= predefined_fm("S25"),
		baseEngine	= predefined_engine("9GSch605"),
	},
	variantSpecificParams = {
		["C_25"] = {
			displayName			= LocStr.S25OFM,
			displayNameShort	= LocStr.dNS.S25,
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("C_25") },
			modelFileName		= "c-25",
			properties			= { dist_min = 1600, dist_max = 3000 },
			paramOverrides		= { wsTypeOfWeaponL4 = C_25 }
		},
		["S-25-O"] = {
			displayName			= LocStr.S25O,
			displayNameShort	= LocStr.dNS.S25,
			schemeName			= "nurs-standard",
			payloadParams		= { warhead = predefined_warhead("S_25_O") },
			modelFileName		= "C-25O",
			properties			= { dist_min = 2000, dist_max = 3000 }
		},
	}
})

BatchDeclareLoadout(
	RLData.O25,
	{
		["{A0648264-4BC0-4EE8-A543-D119F6BA4257}"] = {
			attribute		= { wsType_Weapon, wsType_NURS, wsType_Container, S_25_C },
			displayName		= ("%s - 1 x %s, 340 %s %s"):format(LocStr.O25, cLocStr.UnGdRkts, cLocStr.mm, LocStr.S25OFM),
			Picture			= "S25_OFM.png",
			weaponResName	= "C_25",
		},
		["{TWIN_S25}"] = {
			attribute		= { wsType_Weapon, wsType_NURS, wsType_Container, 181 },
			displayName		= ("2 x %s - 1 x %s, 340 %s %s"):format(LocStr.O25, cLocStr.UnGdRkts, cLocStr.mm, LocStr.S25OFM),
			multiRackParams	= { MER = RLData.Su27TwinPylon, elemCLSID = "{A0648264-4BC0-4EE8-A543-D119F6BA4257}" },
			Picture			= "S25_OFM.png",
			weaponResName	= "C_25"
		},
		["{S_25_O}"] = {
			displayName		= ("%s - 1 x %s, 420 %s %s"):format(LocStr.O25, cLocStr.UnGdRkts, cLocStr.mm, LocStr.S25O),
			Picture			= "S25_O.png",
			weaponResName	= "S-25-O"
		},
		["{TWIN_S25_O}"] = {
			displayName		= ("2 x %s - 1 x %s, 420 %s %s"):format(LocStr.O25, cLocStr.UnGdRkts, cLocStr.mm, LocStr.S25O),
			multiRackParams	= { MER = RLData.Su27TwinPylon, elemCLSID = "{S_25_O}" },
			Picture			= "S25_O.png",
			weaponResName	= "S-25-O"
		},
	},
	S25_Variants
)

--#endregion
