------------------------------------------------------------------------------------
-- This is a little script to help me practice low level helicopter flights in DCS.
-- You can use it to build your own training mission.
--
-- Usage:
--
-- In mission editor:
--
--   1. Go to the triggers panel, and create a new "MISSION START" trigger, with only one "DO SCRIPT FILE" action.
--   2. Click the "OPEN" button just below the "ACTIONS" box, and choose this script file.
--   3. Put a trigger zone anywhere on the map, set its name to "GOAL", and its radius to about 100 feet.
--   4. Put your favorite helicopters on the map, set their skill to "Client" or "Player", and their country to any blue one.
--
-- In the mission:
--
--   1. There should be a message saying "Low Level Flight script loaded."
--   2. You can then start/stop the training in the communication menu "F10. Other..." -> "Low-Level Flight Training".
--   3. A big smoke will be shown on top of the landing zone during the training session.
--   4. After successfully landing in the landing zone, the script will report your total flight time.
--
-- Tips:
--
--   * Altitude limits are AGL.
--   * There will be brutal consequence if you fly too high.
--   * The default altitude limit is 50 feet.
--   * You can change a unit's alt. limit by appending "_<number>" to its name. For example, a unit named "Ninja_30" will have a limit of 30 feet.
--   * You can put a helipad or any other object in the center of the landing zone, to make it more obvious.
--   * In theory, the script can work in multiplayer missions, but I never tested it.
--


---------------------------------------------------------
-- User Variables                                      --
---------------------------------------------------------

local LZ_NAME         = "GOAL"    -- Name of the trigger zone which designates the landinag zone.
local ALTITUDE_LIMIT  = 50        -- In feet. Must fly below this altitude. Altitude limits set in unit names override this value.
local MAX_FOULS       = 5         -- (Roughly) how may seconds you can stay above the altitude limit.


---------------------------------------------------------
-- Global Definitions                                  --
---------------------------------------------------------

local CHECK_INTERVAL      = 1
local EXPLOSION_POWER     = 5

local FEET_PER_METER      = 3.2808399
local DEFAULT_MSG_TIMEOUT = 10
local LZ_SMOKE_NAME       = "LZ_SMOKE"


function msg(content)
    trigger.action.outText(content, DEFAULT_MSG_TIMEOUT)
end


function msg_to_unit(unit_id, content)
    trigger.action.outTextForUnit(unit_id, content, DEFAULT_MSG_TIMEOUT)
end


function calc_distance(pt1, pt2)
    local dx = pt1.x - pt2.x
    local dz = pt1.z - pt2.z
    return math.sqrt(dx * dx + dz * dz)
end


function get_all_blue_client_helicopters()
    local units = coalition.getPlayers(coalition.side.BLUE)
    local heli_units = {}
    for _, u in pairs(units) do
        local desc = u:getDesc()
        if desc.category == Unit.Category.HELICOPTER then
            table.insert(heli_units, u)
        end
    end
    return heli_units
end


---------------------------------------------------------
-- Judge Object                                        --
---------------------------------------------------------

function handleLand(judge, ev)
    if judge:isInProgress() then
        local lz = judge:getLandingZone()
        if calc_distance(ev.initiator:getPoint(), lz.point) < lz.radius then
            judge.time_record = ev.time - judge.start_time
            judge:stop()
            local unit = ev.initiator
            msg(unit:getName() .. " landed successfully! Total flight time: " .. judge.time_record .. " seconds")
        end
    end
end


local Judge = {
    event_handlers    = {
        [world.event.S_EVENT_RUNWAY_TOUCH]   = handleLand,
    },
}


function Judge:onEvent(ev)
    local unit = ev.initiator
    if (not unit) or unit:getName() ~= self.unit_name then
        return
    end

    local handler_func = self.event_handlers[ev.id]
    if handler_func then
        handler_func(self, ev)
    end
end


function Judge:start(time)
    if nil == time then
        return
    end

    self:stop()
    self.start_time = time
    self:scheduleTimer()
    self:showLandingZoneSmoke()
    world.addEventHandler(self)

    local unit = self:getTrackingUnit()
    if unit then
        msg_to_unit(unit:getID(), "YOUR ALTITUDE LIMIT IS " .. self.altitude_limit .. " FEET")
    end
end


function Judge:stop()
    if self:isInProgress() then
        world.removeEventHandler(self)
        self:hideLandingZoneSmoke()
        self:cancelTimer()
        self.start_time = nil
    end
end


function Judge:isInProgress()
    return nil ~= self.start_time
end


function Judge:getTrackingUnit()
    local unit = Unit.getByName(self.unit_name)
    if (not unit) or (not unit:isActive()) or (unit:getLife() < 1) then
        return nil
    end
    return unit
end


function Judge:getLandingZone()
    return trigger.misc.getZone(self.landing_zone_name)
end


function Judge:checkAltitude()
    local unit = self:getTrackingUnit()
    if not unit then
        return nil
    end

    local pos = unit:getPoint()
    local gnd_lv = land.getHeight({x = pos.x, y = pos.z})
    local agl = pos.y - gnd_lv

    return ((agl * FEET_PER_METER) <= self.altitude_limit)
end


function Judge:doJudge(time)
    local result = self:checkAltitude()
    if nil == result then
        msg("Unit deactivated. Stopping the training task for " .. self.unit_name .. ".")
        self:stop()
        return nil
    elseif result then
        self.fouls = 0
    else
        local unit = self:getTrackingUnit()
        self.fouls = self.fouls + 1
        msg_to_unit(unit:getID(), "YOU ARE FLYING TOO HIGH! Fouls = " .. self.fouls)
        if self.fouls > self.max_fouls then
            trigger.action.explosion(unit:getPoint(), self.tnt_equivalent)
        end
    end

    return time + self.check_interval
end


function Judge:cancelTimer()
    if self.timer_id then
        timer.removeFunction(self.timer_id)
        self.timer_id = nil
    end
end


function Judge:scheduleTimer()
    self:cancelTimer()
    local time = timer.getTime()
    self.timer_id = timer.scheduleFunction(self.doJudge, self, time + self.check_interval)
end


function Judge:showLandingZoneSmoke()
    local lz = self:getLandingZone()
    if nil == lz then
        msg("Trigger zone not found: " .. self.landing_zone_name)
        return
    end

    local lz_pos = lz.point
    lz_pos.y = land.getHeight({x = lz_pos.x, y = lz_pos.z}) + 45
    self.smoke_name = LZ_SMOKE_NAME .. "_" .. self.unit_name
    trigger.action.effectSmokeBig(lz_pos, 8, 1, self.smoke_name)
end


function Judge:hideLandingZoneSmoke()
    if nil ~= self.smoke_name then
        trigger.action.effectSmokeStop(self.smoke_name)
        self.smoke_name = nil
    end
end


function Judge:new(u_name, lz_name, alt_limit, max_fouls)
    local judge = {
        unit_name         = u_name,
        landing_zone_name = lz_name,
        altitude_limit    = alt_limit,  -- IN FEET
        max_fouls         = max_fouls,
        fouls             = 0,
        check_interval    = CHECK_INTERVAL,
        timer_id          = nil,
        tnt_equivalent    = EXPLOSION_POWER,
        smoke_name        = nil,
    }
    setmetatable(judge, {__index = self})
    return judge
end


------------------------------------------------------------
-- Communication Menu Items                               --
------------------------------------------------------------

local menu_states = {
    judges         = nil,
    start_timer_id = nil,
}


function commMenuStop()
    if menu_states.start_timer_id then
        timer.removeFunction(menu_states.start_timer_id)
        menu_states.start_timer_id = nil
        msg("TRAINING ABORTED.")
    end

    if nil == menu_states.judges then
        return
    end

    for _, j in pairs(menu_states.judges) do
        j:stop()
    end

    menu_states.judges = nil

    msg("TRAINING ABORTED.")
end


function startTimer(sec, time)
    if sec > 0 then
        msg("Training starting in: " .. sec)
        menu_states.start_timer_id = timer.scheduleFunction(startTimer, sec - 1, time + 1)
    else
        menu_states.start_timer_id = nil
        menu_states.judges = {}
        local time = timer.getTime()
        local client_units = get_all_blue_client_helicopters()
        for _, u in pairs(client_units) do
            local name = u:getName()
            local limit_in_name = string.match(name, "^.*_(%d+)$")
            local alt_limit = ALTITUDE_LIMIT
            if nil ~= limit_in_name then
                alt_limit = tonumber(limit_in_name)
            end
            local j = Judge:new(name, LZ_NAME, alt_limit, MAX_FOULS)
            j:start(time)
            table.insert(menu_states.judges, j)
        end

        msg("TRAINING STARTED!")
    end

    return nil
end


function commMenuStart()
    commMenuStop()
    startTimer(5, timer.getTime())
end


function addCommMenu()
    local menu = missionCommands.addSubMenu("Low-Level Flight Training")
    missionCommands.addCommand("Start", menu, commMenuStart)
    missionCommands.addCommand("Stop",  menu, commMenuStop)
end


------------------------------------------------------------
-- Initialization Code                                    --
------------------------------------------------------------

addCommMenu()
msg("Low Level Flight script loaded.")
