--[[===========================================================================
  Mortar_Deploy_AI_static_v2.9.lua  (FULL DROP-IN — minimal change)
  - EXACTLY your v2.8 behavior preserved.
  - Only change: each team uses its own logistics helo:
      LZ1 -> MTR_LOGI_1, LZ2 -> MTR_LOGI_2, LZ3 -> MTR_LOGI_3
  - All other logic (markers, persistence, menus, timing, wording) unchanged.
===========================================================================]]--

trigger.action.outText(">>> Mortar_Deploy_AI_static_v2.8: booting…", 10)

-- ===== Requirements =====
if not (SPAWN and COORDINATE and AIRBASE and SCHEDULER and TIMER) then
  trigger.action.outText("ERROR: MOOSE not detected. Load MOOSE.lua BEFORE this script.", 25)
  return
end
if _SETTINGS and _SETTINGS.SetPlayerMenuOn then _SETTINGS:SetPlayerMenuOn() end

-- ===== Config =====
local SCRIPT_NAME, SCRIPT_VERSION = "Mortar_Deploy_AI_static_v2.8.lua", "2.8"
local AIRBASE_NAME      = "Akrotiri"

-- Teams (fixed, static menus)
local TEAM_CODES        = { "LZ1", "LZ2", "LZ3" }  -- maps to markers MTR1..MTR3

-- Flight profile
local CRUISE_KTS        = 90
local FINAL_KTS         = 35
local APPROACH_SLOW_M   = 1200
local LZ_NEAR_M         = 300
local LAND_TASK_DELAY_S = 8
local LZ_HOLD_S         = 25

-- Ground behavior / timing
local SPAWN_OFFSET_M    = 20
local RESUPPLY_HOLD_S   = 45
local DESPAWN_AFTER_RTB = 300
local RTB_TOUCH_NEAR_M  = 600

-- Enemy / targeting
local ENEMY_SIDE        = coalition.side.RED
local RECCE_RADIUS_M    = 5000
local ENGAGE_ZONE_M     = 6000

-- FireAtPoint behavior
local FIRE_RADIUS_M     = 30
local RETASK_ENABLED    = true
local RETASK_INTERVAL_S = 60

-- Rearm pool
local REARMS_PER_POOL   = 2
local REARM_TIME_S      = 20

-- ==== ORIGINAL line kept (do not remove) ====
-- AI logistics helo to reset the pool when 0
local LOGI_TEMPLATE     = "MTR_LOGI"  -- BLUE, Late Activated, Uncontrolled
local LOGI_LAND_HOLD_S  = 20
local LOGI_NEAR_M       = 120

-- ==== MINIMAL ADD: per-team template map (NEW) ====
local LOGI_TEMPLATE_BY_LZ = {
  LZ1 = "MTR_LOGI_1",
  LZ2 = "MTR_LOGI_2",
  LZ3 = "MTR_LOGI_3",
}

-- Crew animation preference (true keeps ALARM AUTO; helps crew stay standing)
local KEEP_CREW_STANDING = false

-- Markers
local MARKER_TAG_PREFIX = "MTR"

-- Templates
local INSERT_TEMPLATES  = { LZ1="MTR_HELO_LZ1", LZ2="MTR_HELO_LZ2", LZ3="MTR_HELO_LZ3" }
local EXTRACT_TEMPLATES = { LZ1="MTR_REC_LZ1",  LZ2="MTR_REC_LZ2",  LZ3="MTR_REC_LZ3"  }
local TUBES_TEMPLATE    = "Mortar"

-- ===== Persistence (CSV) =====
local SAVE_DIR  = "C:/Users/Aidi/Saved Games/DCS/Missions/Saves/" --<<<<<< Change to your username C:/Users/<your_name>/Saved_Games....>>>>>>
local SAVE_FILE = SAVE_DIR .. "Mortar_AI_Teams.csv"

local function ensureDir(path)
  local dir = path:match("^(.*)/[^/]+$"); if not dir then return end
  local seg=""; for part in dir:gmatch("[^/]+") do
    seg = (seg=="" and part or (seg.."/"..part))
    pcall(function() lfs.mkdir(seg) end)
  end
end

local function _csvEscape(s) if not s then return "" end; s=tostring(s); if s:find("[,\"]") then s='"'..s:gsub('"','""')..'"' end; return s end
local function _csvSplit(line)
  local out, i, inq, buf = {}, 1, false, {}
  while i <= #line do
    local c = line:sub(i,i)
    if inq then
      if c == '"' and line:sub(i+1,i+1) == '"' then buf[#buf+1] = '"'; i=i+2
      elseif c == '"' then inq=false; i=i+1
      else buf[#buf+1]=c; i=i+1 end
    else
      if c == '"' then inq=true; i=i+1
      elseif c == ',' then out[#out+1]=table.concat(buf); buf={}; i=i+1
      else buf[#buf+1]=c; i=i+1 end
    end
  end
  out[#out+1]=table.concat(buf)
  return out
end

-- ===== Utilities =====
local function banner(msg,t) trigger.action.outText(msg, t or 8) end
local function km(m) return (m or 0) / 1000 end
local function vec2(c) local v=c:GetVec2(); return {x=v.x,y=v.y} end

local function isGroupAliveByName(n)
  local g=Group.getByName(n); if not g then return false end
  for _,u in ipairs(g:getUnits() or {}) do
    if u and u:isExist() and (u.getLife and u:getLife()>0) then return true end
  end
  return false
end

-- Try to read current team heading in degrees
local function _teamHeadingDeg(code, fallback)
  local name = _G.lastTEAMAnchorByLZ and lastTEAMAnchorByLZ[code]
  if not name then return fallback or 0 end

  if GROUP and GROUP.FindByName then
    local mg = GROUP:FindByName(name)
    if mg then
      local mu = mg:GetUnit(1)
      if mu and mu.IsAlive and mu:IsAlive() and mu.GetHeading then
        local ok,h = pcall(function() return mu:GetHeading() end)
        if ok and h then return h end
      end
    end
  end

  local g = Group.getByName(name); if not g then return fallback or 0 end
  local u = g:getUnit(1); if not u then return fallback or 0 end
  local pos = u:getPosition(); if not pos or not pos.x then return fallback or 0 end
  local rad = math.atan2(pos.x.z, pos.x.x)
  return (math.deg(rad)) % 360
end

-- ===== Ammo crates (spawn + despawn) =====
local _MTR_CARGO_TYPES = {
  "ammo_cargo","ammo_box_cargo","box_cargo","container_cargo","barrels_cargo","f_bar_cargo"
}
local MTR_PROPS_BACK_DIST_M = 10
local MTR_PROPS_RADIUS_M    = 2

local function _mtrCountryIdFromTeam(code)
  local anchor = lastTEAMAnchorByLZ and lastTEAMAnchorByLZ[code]
  if anchor then
    local g = Group.getByName(anchor)
    if g then
      local u = g:getUnit(1)
      if u then
        local ok,id = pcall(function() return u:getCountry() end)
        if ok and id then return id end
      end
    end
  end
  if rawget(_G,"country") and country.id then
    return country.id.CJTF_BLUE or country.id.USA or 2
  end
  return 2
end

local function _mtrSpawnOneStatic(cid, name, x, z)
  for _,typ in ipairs(_MTR_CARGO_TYPES) do
    local static = { category = "Cargos", type = typ, name = name .. "_" .. typ, heading = 0, x = x, y = z, canCargo = false }
    local ok = pcall(coalition.addStaticObject, cid, static)
    if ok then return true end
  end
  return false
end

function spawnAmmoPropsForTeam(code, coord, count)
  if not coord then return end
  local cid = _mtrCountryIdFromTeam(code)
  local hdg  = (teamHdgByLZ and teamHdgByLZ[code]) or 0
  local back = (hdg + 180) % 360
  local center = coord:Translate(MTR_PROPS_BACK_DIST_M, back)
  local n = math.max(1, tonumber(count or 3) or 3)

  -- Find a nearby point with the lowest local slope so statics don't "float".
  local function flattestNear(baseC, searchR, probes)
    local bestC, bestScore
    local function slopeScore(c)
      -- sample ground height at the point and tiny offsets; sum diffs as a slope proxy
      local h  = c:GetLandHeight()
      local dx = COORDINATE:New(c.x + 1, 0, c.z); local hx = dx:GetLandHeight()
      local dz = COORDINATE:New(c.x, 0, c.z + 1); local hz = dz:GetLandHeight()
      local d1 = math.abs(hx - h) + math.abs(hz - h)
      return d1
    end
    for i=1,(probes or 12) do
      local ang = (i * (360 / (probes or 12))) % 360
      local cand = baseC:Translate(searchR or 1.5, ang)
      local s = slopeScore(cand)
      if not bestScore or s < bestScore then bestScore, bestC = s, cand end
    end
    -- also consider the original point
    local s0 = slopeScore(baseC)
    if not bestScore or s0 < bestScore then return baseC end
    return bestC or baseC
  end

  -- Build nominal positions (unchanged pattern).
  local positions = {}
  if n >= 1 then positions[#positions+1] = center end
  if n >= 2 then positions[#positions+1] = center:Translate(MTR_PROPS_RADIUS_M, (back + 30) % 360) end
  if n >= 3 then positions[#positions+1] = center:Translate(MTR_PROPS_RADIUS_M, (back - 30) % 360) end
  for i = 4, n do
    local ang = (back + math.random(-75, 75)) % 360
    local r   = math.random(math.floor(MTR_PROPS_RADIUS_M/2), MTR_PROPS_RADIUS_M)
    positions[#positions+1] = center:Translate(r, ang)
  end

  -- For each position, nudge to the flattest nearby micro-spot, then spawn.
  for i,pt in ipairs(positions) do
    local flat = flattestNear(pt, 1.5, 12) -- ~1.5m ring, 12 probes
    local p = flat:GetVec3()
    local name = string.format("MTR_AMMO_%s_%02d", code, i)
    local ok = _mtrSpawnOneStatic(cid, name, p.x, p.z)
    if (not ok) and env and env.info then
      env.info(string.format("[MTR] Static ammo spawn failed at (%.1f, %.1f)", p.x, p.z))
    end
  end
end


function removeAmmoPropsForTeam(code)
  local prefix = "MTR_AMMO_"..tostring(code).."_"
  for _, side in ipairs({ coalition.side.BLUE, coalition.side.RED, coalition.side.NEUTRAL }) do
    local statics = coalition.getStaticObjects(side) or {}
    for _, s in ipairs(statics) do
      if s and s.getName then
        local name = s:getName() or ""
        if name:sub(1, #prefix) == prefix then pcall(function() s:destroy() end) end
      end
    end
  end
end

-- ===== Spawners =====
local InsertSPN, ExtractSPN = {}, {}
for code,tpl in pairs(INSERT_TEMPLATES)  do InsertSPN[code]  = SPAWN:New(tpl) end
for code,tpl in pairs(EXTRACT_TEMPLATES) do
  local s = SPAWN:New(tpl); if s.InitUncontrolled then s:InitUncontrolled(true) end
  ExtractSPN[code] = s
end
local tubesSPN = SPAWN:New(TUBES_TEMPLATE)

-- Logistics helo spawner (ORIGINAL, kept)
local LogiSPN = LOGI_TEMPLATE and SPAWN:New(LOGI_TEMPLATE) or nil
if LogiSPN and LogiSPN.InitUncontrolled then LogiSPN:InitUncontrolled(true) end

-- ===== State =====
local activeInsert   = { LZ1=false,LZ2=false,LZ3=false }
local insertHelos    = { LZ1=nil,  LZ2=nil,  LZ3=nil   }
local activeExtract  = { LZ1=false,LZ2=false,LZ3=false }
local extractHelos   = { LZ1=nil,  LZ2=nil,  LZ3=nil   }

lastTEAMAnchorByLZ       = { LZ1=nil,LZ2=nil,LZ3=nil }  -- mortar group name
local rearmsRemainingByLZ= { LZ1=nil,LZ2=nil,LZ3=nil }  -- int

-- Persisted placement/heading per LZ
teamCoordByLZ            = { LZ1=nil,LZ2=nil,LZ3=nil }  -- COORDINATE
teamHdgByLZ              = { LZ1=0,  LZ2=0,  LZ3=0   }  -- degrees

-- Map markers (read-only) for deployed teams
local teamMarkerIdByLZ   = { LZ1=nil,LZ2=nil,LZ3=nil }  -- int marker id
local _nextMTRMarkerId   = 98000                        -- unique id counter for markers


-- Logistics dispatch/track per LZ
local logiActiveByLZ     = { LZ1=false,LZ2=false,LZ3=false }
local logiHelos          = { LZ1=nil,  LZ2=nil,  LZ3=nil   }

-- Ammo watchers (per LZ)
local ammoWatchByLZ            = { LZ1=nil,LZ2=nil,LZ3=nil }
local ammoDepletedNotifiedByLZ = { LZ1=false,LZ2=false,LZ3=false }

-- Team menu + status line handles
local _TeamMenuByLZ       = { LZ1=nil,LZ2=nil,LZ3=nil }
local _StatusLineByLZ     = { LZ1=nil,LZ2=nil,LZ3=nil }

-- ===== Recce / Threat helpers =====
local function countTargetsInRange(anchorName)
  local g=Group.getByName(anchorName); if not g then return 0 end
  local u=g:getUnit(1); if not (u and u:isExist()) then return 0 end
  local p=u:getPoint(); if not p then return 0 end
  local A=COORDINATE:New(p.x,0,p.z)
  local n=0
  for _,eg in ipairs(coalition.getGroups(ENEMY_SIDE, Group.Category.GROUND) or {}) do
    for _,eu in ipairs(eg:getUnits() or {}) do
      if eu and eu:isExist() then
        local ep=eu:getPoint(); if ep then
          local d = A:Get2DDistance(COORDINATE:New(ep.x,0,ep.z))
          if d <= RECCE_RADIUS_M then n=n+1 end
        end
      end
    end
  end
  return n
end

local function runTeamRecceOnce(anchorName)
  local g=Group.getByName(anchorName); if not g then banner("Threats: team not found",6); return end
  local u=g:getUnit(1); if not (u and u:isExist()) then banner("Threats: team not present",6); return end
  local p=u:getPoint(); if not p then banner("Threats: team not present",6); return end
  local A=COORDINATE:New(p.x,0,p.z)
  local items={}
  for _,eg in ipairs(coalition.getGroups(ENEMY_SIDE, Group.Category.GROUND) or {}) do
    local gn=eg.getName and eg:getName() or "?"
    for _,eu in ipairs(eg:getUnits() or {}) do
      if eu and eu:isExist() and eu.getTypeName then
        local ep=eu:getPoint(); if ep then
          local d=A:Get2DDistance(COORDINATE:New(ep.x,0,ep.z))
          if d<=RECCE_RADIUS_M then items[#items+1]=string.format("%s (%s) — %.1f Km", eu:getTypeName(), gn, km(d)) end
        end
      end
    end
  end
  if #items>0 then banner("MTR Threats:\n"..table.concat(items,"\n"),12) else banner("MTR Threats: Nothing in range.",8) end
end

-- ===== ROE / Tasks =====
local function setROEHold(name)
  local g = Group.getByName(name); if not g then return end
  local c = g:getController()
  if c and AI and AI.Option and AI.Option.Ground then
    pcall(c.setOption, c, AI.Option.Ground.id.ROE,         AI.Option.Ground.val.ROE.WEAPON_HOLD)
    pcall(c.setOption, c, AI.Option.Ground.id.ALARM_STATE, AI.Option.Ground.val.ALARM_STATE.GREEN)
  end
end
local function setROEFree(name, alarmToRed)
  local g = Group.getByName(name); if not g then return end
  local c = g:getController()
  if c and AI and AI.Option and AI.Option.Ground then
    pcall(c.setOption, c, AI.Option.Ground.id.ROE, AI.Option.Ground.val.ROE.WEAPON_FREE)
    if alarmToRed then
      pcall(c.setOption, c, AI.Option.Ground.id.ALARM_STATE, AI.Option.Ground.val.ALARM_STATE.RED)
    else
      pcall(c.setOption, c, AI.Option.Ground.id.ALARM_STATE, AI.Option.Ground.val.ALARM_STATE.AUTO)
    end
  end
end

local function pushEngageZoneTask(groupName, centerCoord, radius)
  if not groupName or not centerCoord then return end
  local g = Group.getByName(groupName); if not g then return end
  local c = g:getController(); if not c then return end
  local v2 = centerCoord:GetVec2()
  local task = {
    id = 'EngageTargetsInZone',
    params = {
      targetTypes = { 'Infantry', 'Armor', 'Artillery', 'Air Defence', 'Fortifications', 'Ground Units' },
      x = v2.x, y = v2.y, zoneRadius = radius or ENGAGE_ZONE_M, priority = 0, weaponType = 0,
    }
  }
  pcall(function() c:pushTask(task) end)
end

local function _nearestEnemyVec2(centerC, maxR)
  local best, bestD = nil, 1e12
  for _,eg in ipairs(coalition.getGroups(ENEMY_SIDE, Group.Category.GROUND) or {}) do
    for _,eu in ipairs(eg:getUnits() or {}) do
      if eu and eu:isExist() then
        local ep = eu:getPoint()
        if ep then
          local d = centerC:Get2DDistance(COORDINATE:New(ep.x,0,ep.z))
          if d < bestD and d <= (maxR or ENGAGE_ZONE_M) then
            bestD = d; best = { x = ep.x, y = ep.z }
          end
        end
      end
    end
  end
  return best
end

local function _orderFireAtPoint(groupName, tgtV2, radiusM)
  local g = Group.getByName(groupName); if not g then return false end
  local c = g:getController();          if not c then return false end
  local task = { id = 'FireAtPoint', params = { point = tgtV2, radius = radiusM or FIRE_RADIUS_M } }
  pcall(function() c:pushTask(task) end)
  return true
end

-- Go hot + optional retask loop
local function _goHot(tubesName, coord)
  setROEFree(tubesName, not KEEP_CREW_STANDING)
  local tgtV2 = _nearestEnemyVec2(coord, ENGAGE_ZONE_M)
  if tgtV2 then _orderFireAtPoint(tubesName, tgtV2, FIRE_RADIUS_M) else pushEngageZoneTask(tubesName, coord, ENGAGE_ZONE_M) end
  if RETASK_ENABLED then
    local aliveName = tubesName
    SCHEDULER:New(nil, function()
      if not isGroupAliveByName(aliveName) then return end
      local v2 = _nearestEnemyVec2(coord, ENGAGE_ZONE_M)
      if v2 then _orderFireAtPoint(aliveName, v2, FIRE_RADIUS_M) end
    end, {}, RETASK_INTERVAL_S, RETASK_INTERVAL_S)
  end
end

-- ===== Ammo watcher =====
local function _groupShellsRemaining(groupName)
  local g = Group.getByName(groupName); if not g then return 0 end
  local total = 0
  for _,u in ipairs(g:getUnits() or {}) do
    if u and u:isExist() and u.getAmmo then
      local ammo = u:getAmmo()
      if ammo then
        for _,w in ipairs(ammo) do
          local count = (w and (w.count or w.ammo)) or 0
          local cat   = w and w.desc and w.desc.category or nil
          if count and count > 0 and cat == 3 then total = total + count end -- shells
        end
      end
    end
  end
  return total
end


local function _startAmmoWatch(code)
  local anchor = lastTEAMAnchorByLZ[code]
  if not anchor then return end

  ammoDepletedNotifiedByLZ[code] = false
  if ammoWatchByLZ[code] then
    ammoWatchByLZ[code]:Stop()
    ammoWatchByLZ[code] = nil
  end

  local lastRounds = nil  -- stores previous count to detect change

  local sch
  sch = SCHEDULER:New(nil, function()
    if (not anchor) or (not isGroupAliveByName(anchor)) or (lastTEAMAnchorByLZ[code] ~= anchor) then
      if sch then sch:Stop() end
      ammoWatchByLZ[code] = nil
      return
    end

    local rounds = _groupShellsRemaining(anchor)

    -- first tick: initialise
    if lastRounds == nil then
      lastRounds = rounds
      return
    end

    -- detect transition from >0 to 0 (while still alive)
    if (lastRounds > 0) and (rounds <= 0) and (not ammoDepletedNotifiedByLZ[code]) then
      ammoDepletedNotifiedByLZ[code] = true
      local idx = tonumber(code:match("LZ(%d)")) or 0
      banner(string.format("Team %d Req Reload", idx), 8)
    end

    -- store last seen count
    lastRounds = rounds
  end, {}, 2, 2)

  ammoWatchByLZ[code] = sch
end



local function _stopAmmoWatch(code)
  if ammoWatchByLZ[code] then ammoWatchByLZ[code]:Stop(); ammoWatchByLZ[code]=nil end
  ammoDepletedNotifiedByLZ[code] = false
end

-- ===== Logistics helpers =====
local function _autoDespawnAfterLanding(heloGroup, baseC, delaySec)
  if not (heloGroup and baseC) then return end
  local armed=false
  local sch; sch=SCHEDULER:New(nil,function()
    if not heloGroup or not heloGroup.IsAlive or not heloGroup:IsAlive() then sch:Stop(); return end
    local here=heloGroup:GetCoordinate()
    if here and here.Get2DDistance and here:Get2DDistance(baseC) <= RTB_TOUCH_NEAR_M then
      local u = heloGroup:GetUnit(1)
      if u then
        local d=u:GetDCSObject()
        if d and d.inAir and not d:inAir() then
          if not armed then
            armed=true
            TIMER:New(function() if heloGroup:IsAlive() then pcall(function() heloGroup:Destroy() end) end end):Start(delaySec or DESPAWN_AFTER_RTB)
            sch:Stop()
          end
        end
      end
    end
  end,{},2,2)
end

-- ===== Persistence helpers =====
local function SaveState()
  ensureDir(SAVE_FILE)
  local f = io.open(SAVE_FILE, "w"); if not f then return end
  f:write("slot,state,x,z,hdg,rearms\n")
  for i,code in ipairs(TEAM_CODES) do
    local state = (lastTEAMAnchorByLZ[code] and isGroupAliveByName(lastTEAMAnchorByLZ[code])) and "Deployed" or "Undeployed"
    local coord = teamCoordByLZ[code]
    local x = coord and coord.x or ""
    local z = coord and coord.z or ""
    local hdg = teamHdgByLZ[code] or 0
    local rearms = rearmsRemainingByLZ[code]
    local line = table.concat({
      _csvEscape(code), _csvEscape(state), _csvEscape(x), _csvEscape(z), _csvEscape(hdg), _csvEscape(rearms)
    }, ",")
    f:write(line.."\n")
  end
  f:close()
end

local function _spawnDeployedFromPersist(code, x, z, hdg, rearms)
  local coord = COORDINATE:New(x or 0, 0, z or 0)
  teamCoordByLZ[code] = coord
  teamHdgByLZ[code]   = tonumber(hdg or 0) or 0
  rearmsRemainingByLZ[code] = tonumber(rearms or REARMS_PER_POOL) or REARMS_PER_POOL
  if tubesSPN.InitHeading then tubesSPN:InitHeading(teamHdgByLZ[code]) end
  local tubes = tubesSPN:SpawnFromCoordinate(coord)
  if not tubes or not tubes:IsAlive() then return end
  local name = tubes:GetName()
  lastTEAMAnchorByLZ[code] = name
  setROEHold(name)

  -- Create read-only marker for restored deployment
  do
    _nextMTRMarkerId = (_nextMTRMarkerId or 98000) + 1
    local id   = _nextMTRMarkerId
-- Offset the marker 500 m to the team's right so it doesn’t sit on the mortars
local hdg = teamHdgByLZ[code] or 0
local right = (hdg + 120) % 360
local offsetCoord = coord:Translate(250, right)
local pos3 = offsetCoord:GetVec3()
local idx  = tonumber(code:match("LZ(%d)")) or 0
local text = string.format("MTR%d — Deployed", idx)
pcall(function() trigger.action.markToCoalition(id, text, pos3, coalition.side.BLUE, true) end)
teamMarkerIdByLZ[code] = id
  end

  TIMER:New(function()
    _goHot(name, coord)
    banner(("MTR %s: HOT. Rearms left: %d (restored)"):format(code, rearmsRemainingByLZ[code] or 0), 6)
  end):Start(RESUPPLY_HOLD_S)
  spawnAmmoPropsForTeam(code, coord, 3)
  _startAmmoWatch(code)
end


local function LoadState()
  local f = io.open(SAVE_FILE, "r"); if not f then return end
  local header = f:read("*l")
  for line in f:lines() do
    if line and line:match("%S") then
      local cols = _csvSplit(line)
      local code = (cols[1] or ""):gsub("%s+","")
      local state = (cols[2] or ""):gsub("%s+","")
      local x = tonumber(cols[3] or "")
      local z = tonumber(cols[4] or "")
      local hdg = tonumber(cols[5] or "")
      local rearms = tonumber(cols[6] or "")
      if code == "LZ1" or code == "LZ2" or code == "LZ3" then
        rearmsRemainingByLZ[code] = rearms or REARMS_PER_POOL
        if state == "Deployed" and x and z then
          _spawnDeployedFromPersist(code, x, z, hdg or 0, rearms or REARMS_PER_POOL)
        else
          teamCoordByLZ[code] = nil
          teamHdgByLZ[code]   = hdg or 0
        end
      end
    end
  end
  f:close()
end

-- ===== Spawn mortar team (with ammo crates) =====
local function spawnMortarTeam(lz, code, hdgDeg)
  if not (lz and lz.x and lz.z and code) then return nil,nil end
  local coord = COORDINATE:New(lz.x, 0, lz.z)
  teamCoordByLZ[code] = coord
  teamHdgByLZ[code]   = tonumber(hdgDeg or teamHdgByLZ[code] or 0) or 0
  if tubesSPN.InitHeading then tubesSPN:InitHeading(teamHdgByLZ[code]) end
  local tubes = tubesSPN:SpawnFromCoordinate(coord)
  if not tubes or not tubes:IsAlive() then return nil,nil end
  spawnAmmoPropsForTeam(code, coord, 3)
  local name = tubes:GetName()
  lastTEAMAnchorByLZ[code]   = name
  rearmsRemainingByLZ[code]  = REARMS_PER_POOL
  logiActiveByLZ[code]       = false
  logiHelos[code]            = nil
  setROEHold(name)

  -- Create a read-only coalition marker at the deployed location
  do
    -- allocate a unique id
    _nextMTRMarkerId = (_nextMTRMarkerId or 98000) + 1
    local id   = _nextMTRMarkerId
    local pos3 = coord:GetVec3()
    local idx  = tonumber(code:match("LZ(%d)")) or 0
    local text = string.format("MTR%d — Deployed", idx)
    -- Some DCS builds accept a readOnly bool as 5th param; if ignored, marker is still created.
    pcall(function() trigger.action.markToCoalition(id, text, pos3, coalition.side.BLUE, true) end)
    teamMarkerIdByLZ[code] = id
  end

  TIMER:New(function()
    _goHot(name, coord)
    banner(("MTR %s: HOT. Rearms left: %d"):format(code, rearmsRemainingByLZ[code] or 0), 6)
  end):Start(RESUPPLY_HOLD_S)
  _startAmmoWatch(code)
  SaveState()
  return name
end


-- ===== Extraction =====
local function startRecovery(code)
  if activeExtract[code] then banner(code.." recovery already in progress.",8); return end
  local anchor = lastTEAMAnchorByLZ[code]
  if not (anchor and isGroupAliveByName(anchor)) then banner(code..": No mortar team recorded yet.",8); return end
  local spn = ExtractSPN[code]
  if not spn then banner("No recovery template for "..code,10); return end
  local helo = spn:Spawn()
  if not helo then banner("Recovery spawn failed for "..code,10); return end
  activeExtract[code] = true
  extractHelos[code] = helo
  local idx = tonumber(code:match("LZ(%d)")) or 0
  banner(string.format("Mortar Team %d Extraction Helo coming online", idx), 8)
  TIMER:New(function() if helo and helo:IsAlive() and helo.StartUncontrolled then helo:StartUncontrolled() end end):Start(10)
  local lzCoord, boarded = nil, false
  local smokePopped = false
  local watch
  watch = SCHEDULER:New(nil, function()
    if not helo or not helo:IsAlive() then activeExtract[code] = false extractHelos[code] = nil watch:Stop() return end
    local tg = Group.getByName(anchor); if not tg then activeExtract[code]=false extractHelos[code]=nil watch:Stop() return end
    local tu = tg:getUnit(1); if not (tu and tu:isExist()) then activeExtract[code]=false extractHelos[code]=nil watch:Stop() return end
    local tp = tu:getPoint(); local teamC = COORDINATE:New(tp.x, 0, tp.z)
    local here = helo:GetCoordinate(); local dist = here:Get2DDistance(teamC)
    if (not smokePopped) and dist <= 1000 then smokePopped = true trigger.action.smoke(teamC:GetVec3(), trigger.smokeColor.Blue) banner("Extraction LZ marked with blue smoke.", 6) end
    if not lzCoord then
      helo:RouteToVec2(teamC:GetVec2(), CRUISE_KTS, "Vee")
      if dist <= APPROACH_SLOW_M then
        lzCoord = teamC:Translate(15, 90)
        pcall(function() helo:SetAltitude(lzCoord:GetLandHeight() + 15, true) end)
        helo:RouteToVec2(lzCoord:GetVec2(), FINAL_KTS, "Vee")
        TIMER:New(function()
          if not helo or not helo:IsAlive() then return end
          local DG = helo:GetDCSObject()
          if DG and DG.getController then
            local ctrl = DG:getController()
            if ctrl then local v = lzCoord:GetVec2() pcall(function() ctrl:pushTask({ id='Land', params={ point={ x=v.x, y=v.y }, duration=12 } }) end) end
          end
        end):Start(LAND_TASK_DELAY_S)
      end
      return
    end
    local u1 = helo:GetUnit(1)
    local grounded=false
    if u1 and u1.IsAlive and u1:IsAlive() then local d = u1:GetDCSObject() if d and d.inAir and not d:inAir() then grounded=true end end
    local near = lzCoord and (here:Get2DDistance(lzCoord) <= LZ_NEAR_M) or false
    if (not boarded) and grounded and near then
      boarded = true
	  
	  TIMER:New(function()
  local tubes = Group.getByName(anchor); if tubes then pcall(function() tubes:destroy() end) end
  removeAmmoPropsForTeam(code)
  _stopAmmoWatch(code)

  -- Remove the read-only marker for this team (if present)
  do
    local mid = teamMarkerIdByLZ[code]
    if mid then pcall(function() trigger.action.removeMark(mid) end) end
    teamMarkerIdByLZ[code] = nil
  end

  lastTEAMAnchorByLZ[code] = nil
  teamCoordByLZ[code] = nil
  rearmsRemainingByLZ[code] = rearmsRemainingByLZ[code] or REARMS_PER_POOL
  logiActiveByLZ[code] = false
  logiHelos[code] = nil
  banner("Mortar team recovered", 6)
  SaveState()
  local base = AIRBASE:FindByName(AIRBASE_NAME) or AIRBASE:FindClosestAirbase(helo:GetCoordinate())
  if base then local bc = base:GetCoordinate() helo:RouteToVec2(vec2(bc), CRUISE_KTS, "Vee") _autoDespawnAfterLanding(helo, base:GetCoordinate(), DESPAWN_AFTER_RTB) end
  activeExtract[code] = false extractHelos[code] = nil watch:Stop()
end):Start(10)


    end
  end, {}, 1, 1)
end

-- ===== Rearm (pool; AI resupply when pool is 0) =====
local function _dispatchLogi(code, teamC)
  if logiActiveByLZ[code] then banner(("MTR %s: Logistics already en route."):format(code), 8); return end

  -- ==== THE ONLY FUNCTIONAL CHANGE (per-team template selection) ====
  local tpl = (LOGI_TEMPLATE_BY_LZ and LOGI_TEMPLATE_BY_LZ[code]) or LOGI_TEMPLATE
  local spn = SPAWN:New(tpl)
  if spn and spn.InitUncontrolled then spn:InitUncontrolled(true) end
  local helo = spn:Spawn()
  -- ==== END CHANGE ====

  if not helo then banner(("MTR %s: Logistics spawn failed."):format(code), 10); return end
  logiActiveByLZ[code] = true; logiHelos[code] = helo
  banner(("MTR %s: Logistics helo launched."):format(code), 8)
  TIMER:New(function() if helo and helo:IsAlive() and helo.StartUncontrolled then helo:StartUncontrolled() end end):Start(10)
  local lzCoord=nil; local resupplied=false
  local base = AIRBASE:FindByName(AIRBASE_NAME); local baseC = base and base:GetCoordinate() or nil
  local watch; watch=SCHEDULER:New(nil,function()
    if not helo or not helo:IsAlive() then logiActiveByLZ[code]=false; logiHelos[code]=nil; if watch then watch:Stop() end; return end
    local here = helo:GetCoordinate()
    if not lzCoord then
      helo:RouteToVec2(teamC:GetVec2(), CRUISE_KTS, "Vee")
      if here and here:Get2DDistance(teamC) <= APPROACH_SLOW_M then
        local hdg = teamHdgByLZ[code] or _teamHeadingDeg(code, 0)
        local back = (hdg + 180) % 360
        lzCoord = teamC:Translate(50, back)
        pcall(function() helo:SetAltitude(lzCoord:GetLandHeight()+15, true) end)
        helo:RouteToVec2(lzCoord:GetVec2(), FINAL_KTS, "Vee")
        TIMER:New(function()
          if not helo or not helo:IsAlive() then return end
          local DG=helo:GetDCSObject(); if DG and DG.getController then
            local ctrl = DG:getController()
            if ctrl then local v = lzCoord:GetVec2() pcall(function() ctrl:pushTask({ id='Land', params={ point={ x=v.x, y=v.y }, duration=LOGI_LAND_HOLD_S } }) end) end
          end
        end):Start(LAND_TASK_DELAY_S)
      end
      return
    end
    local near = here and lzCoord and (here:Get2DDistance(lzCoord) <= LOGI_NEAR_M)
    local grounded=false
    local u1 = helo:GetUnit(1)
    if u1 and u1.IsAlive and u1:IsAlive() then local d = u1:GetDCSObject() if d and d.inAir and not d:inAir() then grounded=true end end
    if (not resupplied) and grounded and near then
  resupplied = true
  rearmsRemainingByLZ[code] = REARMS_PER_POOL
banner(("MTR %s: Logistics resupply complete. Rearms reset to %d.\n\n-- Rearm to Re-Engage Targets --"):format(code, REARMS_PER_POOL), 10)
  if baseC then helo:RouteToVec2(vec2(baseC), CRUISE_KTS, "Vee") _autoDespawnAfterLanding(helo, baseC, DESPAWN_AFTER_RTB) end
  logiActiveByLZ[code]=false; logiHelos[code]=nil; if watch then watch:Stop() end
  SaveState()
end

  end, {}, 1, 1)
end

local function rearmMortarTeam(code)
  local anchor = lastTEAMAnchorByLZ[code]
  if not (anchor and isGroupAliveByName(anchor)) then banner(code..": No Mortar team to rearm.",8); return end
  local tg = Group.getByName(anchor); if not tg then banner(code..": Team not found.",8); return end
  local tu = tg:getUnit(1); if not (tu and tu:isExist()) then banner(code..": Team not present.",8); return end
  local tp = tu:getPoint(); if not tp then banner(code..": No position data.",8); return end
  local teamC = COORDINATE:New(tp.x,0,tp.z)
  teamCoordByLZ[code] = teamC

  if rearmsRemainingByLZ[code] == nil then
    rearmsRemainingByLZ[code] = REARMS_PER_POOL
  end

  -- === Only use local stock; no auto helo spawn ===
  if (rearmsRemainingByLZ[code] or 0) <= 0 then
    banner("No local stock - Req Helo Resupply", 10)
    return
  end

  rearmsRemainingByLZ[code] = math.max(0, (rearmsRemainingByLZ[code] or 0) - 1)

  banner(("MTR %s: Rearming... %ds (after rearm: %d left)"):format(code, REARM_TIME_S, rearmsRemainingByLZ[code]), 10)

  local hdgToUse = teamHdgByLZ[code] or _teamHeadingDeg(code, 0)

TIMER:New(function()
  pcall(function() tg:destroy() end)
  if tubesSPN.InitHeading then tubesSPN:InitHeading(hdgToUse) end
  local tubes = tubesSPN:SpawnFromCoordinate(teamC)
  if not tubes or not tubes:IsAlive() then banner(("MTR %s: Rearm failed (spawn)."):format(code), 10); return end
  local newName = tubes:GetName()
  lastTEAMAnchorByLZ[code] = newName
  teamHdgByLZ[code] = hdgToUse
  setROEHold(newName)

  -- 🔸 NEW: Display standby message halfway through the 45s wait (at 25s)
  local idx = tonumber(code:match("LZ(%d)")) or 0
  TIMER:New(function()
    banner(string.format("Mortar Team %d - Standby", idx), 8)
  end):Start(25)

  TIMER:New(function()
    _goHot(newName, teamC)
    banner(("MTR %s: Rearm complete - Ready to Engage | Rearms left: %d"):format(code, rearmsRemainingByLZ[code] or 0), 8)
  end):Start(RESUPPLY_HOLD_S)

  spawnAmmoPropsForTeam(code, teamC, 3)
  _startAmmoWatch(code)
  SaveState()
end):Start(REARM_TIME_S)

end


-- ===== Insert flow =====
local function launchMortarInsert(lz, code)
  if activeInsert[code] then banner(code.." already active.",8); return end
  if lastTEAMAnchorByLZ[code] and isGroupAliveByName(lastTEAMAnchorByLZ[code]) then banner(code.." already deployed.",8); return end
  local sp = InsertSPN[code]; if not sp then banner("Missing insert template for "..code,10); return end
  local helo = sp:Spawn(); if not helo then banner("Insert spawn failed for "..code,10); return end
  activeInsert[code] = true; insertHelos[code] = helo
  banner(("MTR Helo Online from %s"):format(AIRBASE_NAME),8)
  TIMER:New(function() if helo and helo:IsAlive() and helo.StartUncontrolled then helo:StartUncontrolled() end end):Start(10)
  local LZC = COORDINATE:New(lz.x,0,lz.z)
  local lzLocked, deployed = false, false
  local watch; watch = SCHEDULER:New(nil,function()
    if not helo or not helo:IsAlive() then activeInsert[code]=false; insertHelos[code]=nil; watch:Stop(); return end
    local here = helo:GetCoordinate()
    local dist = here and here:Get2DDistance(LZC) or 1e9
    if not lzLocked then
      helo:RouteToVec2(LZC:GetVec2(), CRUISE_KTS, "Vee")
      if dist <= APPROACH_SLOW_M then
        lzLocked = true
        pcall(function() helo:SetAltitude(LZC:GetLandHeight()+15, true) end)
        helo:RouteToVec2(LZC:GetVec2(), FINAL_KTS, "Vee")
        TIMER:New(function()
          if not helo or not helo:IsAlive() then return end
          local DG=helo:GetDCSObject()
          if DG and DG.getController then
            local ctrl=DG:getController()
            if ctrl then
              local v = LZC:GetVec2()
              pcall(function() ctrl:pushTask({ id='Land', params={ point={ x=v.x, y=v.y }, duration=LZ_HOLD_S } }) end)
            end
          end
        end):Start(LAND_TASK_DELAY_S)
      end
      return
    end
    local u1 = helo:GetUnit(1)
    local near = (dist <= LZ_NEAR_M)
    local grounded=false
    if u1 and u1.IsAlive and u1:IsAlive() then local d = u1:GetDCSObject() if d and d.inAir and not d:inAir() then grounded=true end end
    if (not deployed) and grounded and near then
      deployed = true
      TIMER:New(function()
        local hdg = 0
        pcall(function() hdg = (u1.GetHeading and u1:GetHeading()) or (helo.GetHeading and helo:GetHeading()) or 0 end)
        local spawnC = LZC:Translate(SPAWN_OFFSET_M, hdg)
        local v2 = spawnC:GetVec2()
        spawnMortarTeam({ x=v2.x, z=v2.y }, code, hdg)
        banner(("MTR %s: Mortar team on the ground."):format(code),8)
        local base = AIRBASE:FindByName(AIRBASE_NAME) or AIRBASE:FindClosestAirbase(helo:GetCoordinate())
        if base then local bc = base:GetCoordinate() helo:RouteToVec2(vec2(bc), CRUISE_KTS, "Vee") _autoDespawnAfterLanding(helo, bc, DESPAWN_AFTER_RTB) end
        activeInsert[code] = false; insertHelos[code] = nil; watch:Stop()
      end):Start(8)
    end
  end, {}, 1, 1)
end

-- ===== Marker handling (MTR1..MTR3) =====
local _processedMarkers = {}
local function handleMarker(idx,pos,text)
  local txt = (text or ""):upper():gsub("%s+","")
  local n = txt:match("^"..MARKER_TAG_PREFIX.."([1-3])$")
  if not n or _processedMarkers[idx] then return end
  _processedMarkers[idx] = true
  local code = "LZ"..n
  banner(("Marker %s accepted at (%.0f, %.0f)"):format(code, pos.x, pos.z),6)
  launchMortarInsert({ x=pos.x, z=pos.z }, code)
end
do
  if _G.__MTR_EH then pcall(world.removeEventHandler, _G.__MTR_EH) end
  local EH = {}
  function EH:onEvent(e)
    if not e or not e.id then return end
    if e.id == world.event.S_EVENT_MARK_ADDED or e.id == world.event.S_EVENT_MARK_CHANGE then
      if e.pos and e.idx then handleMarker(e.idx, e.pos, e.text or "") end
    end
  end
  world.addEventHandler(EH)
  _G.__MTR_EH = EH
end

-- ===== Static F10 Menus (BLUE) =====
local function _fmtTeamLine(idx, code)
  local anchor = lastTEAMAnchorByLZ[code]
  local isActive = (anchor and isGroupAliveByName(anchor)) or false
  local ammo = rearmsRemainingByLZ[code]; if ammo == nil then ammo = "-" end
  local tgtCount = (isActive and countTargetsInRange(anchor)) or 0
  local logi = logiActiveByLZ[code] and " (LOGI en route)" or ""
  if isActive then
    return string.format("MTR%d: ACTIVE | Rearms:%s | Targets:%d%s", idx, tostring(ammo), tgtCount, logi)
  else
    return string.format("MTR%d: NOT DEPLOYED | Rearms:%s", idx, tostring(ammo))
  end
end

local function AITeams_FieldStatus()
  local lines = { "--- AI Mortar Field Status ---" }
  for i,code in ipairs(TEAM_CODES) do lines[#lines+1] = _fmtTeamLine(i, code) end
  banner(table.concat(lines, "\n"), 12)
end

local function _isDeployed(code)
  local anchor = lastTEAMAnchorByLZ[code]
  return (anchor and isGroupAliveByName(anchor)) or false
end

function _refreshTeamStatusItem(code)
  local parent = _TeamMenuByLZ[code]; if not parent then return end
  local existing = _StatusLineByLZ[code]
  if existing then pcall(missionCommands.removeItemForCoalition, coalition.side.BLUE, existing) end
  local idx = tonumber(code:match("LZ(%d)")) or 0
  local text = _isDeployed(code) and ("Status: Deployed (Team "..idx..")") or ("Status: Undeployed (Team "..idx..")")
  _StatusLineByLZ[code] = missionCommands.addCommandForCoalition(
    coalition.side.BLUE, text, parent,
    function()
      local msg = _isDeployed(code) and "Deployed" or "Undeployed"
      local anchor = lastTEAMAnchorByLZ[code]
      if anchor and isGroupAliveByName(anchor) then
        local g=Group.getByName(anchor); local u=g and g:getUnit(1); local p=u and u:getPoint()
        if p then banner(string.format("Team %d is %s. Coords: %.0f / %.0f", idx, msg, p.x, p.z), 8); return end
      end
      banner(string.format("Team %d is %s.", idx, msg), 6)
    end
  )
end

local function AITeams_Rearm(code)  rearmMortarTeam(code) end
local function AITeams_CallLogi(code)
  local anchor = lastTEAMAnchorByLZ[code]
  if not (anchor and isGroupAliveByName(anchor)) then banner(code..": No team to resupply.",8); return end
  local g = Group.getByName(anchor); local u = g and g:getUnit(1) or nil; local p = u and u:getPoint() or nil
  if not p then banner(code..": Team position unknown.",8); return end
  _dispatchLogi(code, COORDINATE:New(p.x,0,p.z))
end
local function AITeams_Extract(code) startRecovery(code) end
local function AITeams_AbortInsert(code)
  local h = insertHelos[code]
  if h and h:IsAlive() then
    local base = AIRBASE:FindByName(AIRBASE_NAME) or AIRBASE:FindClosestAirbase(h:GetCoordinate())
    if base then local bc = base:GetCoordinate(); h:RouteToVec2(vec2(bc), CRUISE_KTS, "Vee"); _autoDespawnAfterLanding(h, bc, DESPAWN_AFTER_RTB) end
    banner(code..": Insertion aborted.",8)
  else banner(code..": No active insertion.",6) end
end
local function AITeams_Threats(code)
  local n = lastTEAMAnchorByLZ[code]; if n then runTeamRecceOnce(n) else banner(code..": No team.",6) end
end

local function BuildStaticAIMortarMenus()
  local root = missionCommands.addSubMenuForCoalition(coalition.side.BLUE, "AI Mortar Teams")
  for i,code in ipairs(TEAM_CODES) do
    local label = string.format("Team %d (%s)", i, code:gsub("^LZ(%d+)","MTR%1"))
    local m = missionCommands.addSubMenuForCoalition(coalition.side.BLUE, label, root)
    _TeamMenuByLZ[code] = m
    _refreshTeamStatusItem(code)
    missionCommands.addCommandForCoalition(coalition.side.BLUE, "Rearm (Local Stock)", m, function() AITeams_Rearm(code) end)
    missionCommands.addCommandForCoalition(coalition.side.BLUE, "Ammo Resupply",       m, function() AITeams_CallLogi(code) end)
    missionCommands.addCommandForCoalition(coalition.side.BLUE, "Extract (recover)",   m, function() AITeams_Extract(code) end)
    missionCommands.addCommandForCoalition(coalition.side.BLUE, "Abort Insertion",     m, function() AITeams_AbortInsert(code) end)
    missionCommands.addCommandForCoalition(coalition.side.BLUE, "Threats",             m, function() AITeams_Threats(code) end)
  end
  missionCommands.addCommandForCoalition(coalition.side.BLUE, "Field Status", root, AITeams_FieldStatus)
end

BuildStaticAIMortarMenus()

-- ===== Init =====
SCHEDULER:New(nil, LoadState, {}, 1)




banner(SCRIPT_NAME.." v"..SCRIPT_VERSION.." loaded (static menus: MTR1..MTR3, CSV persist, smoke cue, ammo props + despawn on extract).", 7)
