-- WW2PacCVOps.lua
-- DCS World + MOOSE mission script

local WW2PacCVOps = {}

local function _log(msg)
  if env and env.info then
    env.info("[WW2PacCVOps] " .. tostring(msg))
  end
end

local function _warn(msg)
  if env and env.warning then
    env.warning("[WW2PacCVOps] " .. tostring(msg))
  else
    _log("WARNING: " .. tostring(msg))
  end
end

local function _err(msg)
  if env and env.error then
    env.error("[WW2PacCVOps] " .. tostring(msg))
  else
    _log("ERROR: " .. tostring(msg))
  end
end

local function _startsWith(s, prefix)
  if type(s) ~= "string" or type(prefix) ~= "string" then return false end
  return s:sub(1, #prefix) == prefix
end

local function _safeCall(f, ...)
  local ok, res = pcall(f, ...)
  if not ok then
    return false, res
  end
  return true, res
end

local function _templateGroupExists(templateName)
  if type(templateName) ~= "string" then return false end

  if GROUP and GROUP.FindByName then
    local ok, grp = _safeCall(function()
      return GROUP:FindByName(templateName)
    end)
    if ok and grp ~= nil then
      return true
    end
  end

  if Group and Group.getByName then
    local ok, grp = _safeCall(Group.getByName, templateName)
    if ok and grp ~= nil then
      return true
    end
  end

  return false
end

local function _getAirbaseNameFromPlace(place)
  if not place then return nil end
  if type(place) == "table" then
    if place.getName then
      local ok, n = _safeCall(place.getName, place)
      if ok then return n end
    end
    if place.GetName then
      local ok, n = _safeCall(place.GetName, place)
      if ok then return n end
    end
  end
  return nil
end

local function _setAirbaseCoalition(airbaseWrapper, coalitionSide)
  if not airbaseWrapper then return end

  local dcsAirbase = nil
  if airbaseWrapper.GetDCSObject then
    local ok, ab = _safeCall(airbaseWrapper.GetDCSObject, airbaseWrapper)
    if ok then dcsAirbase = ab end
  end

  if dcsAirbase and dcsAirbase.setCoalition then
    _safeCall(dcsAirbase.setCoalition, dcsAirbase, coalitionSide)
    return
  end

  if Airbase and Airbase.getByName then
    local ok, ab = _safeCall(Airbase.getByName, airbaseWrapper:GetName())
    if ok and ab and ab.setCoalition then
      _safeCall(ab.setCoalition, ab, coalitionSide)
    end
  end
end

local function _makeSpawner(templateName)
  if not SPAWN or not SPAWN.New then
    _err("MOOSE SPAWN class not available. Ensure MOOSE is loaded before WW2PacCVOps.lua")
    return nil
  end

  _log(string.format(
    "Template '%s': exists=%s (late-activated template groups must exist in ME)",
    tostring(templateName),
    tostring(_templateGroupExists(templateName))
  ))

  local ok, sp = _safeCall(function()
    return SPAWN:New(templateName)
  end)

  if not ok then
    _warn("Could not create SPAWN from template '" .. tostring(templateName) .. "': " .. tostring(sp))
    return nil
  end

  if sp == nil then
    _warn("SPAWN:New returned nil for template '" .. tostring(templateName) .. "'")
  else
    _log("Spawner created for template '" .. tostring(templateName) .. "'")
  end

  return sp
end

local function _safeSpawn(spawner)
  if not spawner then 
    _warn("SPAWN handle is nil; cannot spawn group")
    return nil 
  end
  local ok, grp = _safeCall(function()
    return spawner:Spawn()
  end)
  if not ok then
    _warn("Spawn failed: " .. tostring(grp))
    return nil
  end
  return grp
end

local function _safeSpawnAtAirbase(spawner, airbase, takeoff)
  if not spawner then return nil end
  if not airbase then
    _warn("SpawnAtAirbase requested but AIRBASE handle is nil; falling back to Spawn()")
    return _safeSpawn(spawner)
  end

  if spawner.SpawnAtAirbase then
    local ok, grp = _safeCall(function()
      return spawner:SpawnAtAirbase(airbase, takeoff or SPAWN.Takeoff.Hot)
    end)
    if not ok then
      _warn("SpawnAtAirbase failed: " .. tostring(grp))
      return nil
    end
    return grp
  end
  return _safeSpawn(spawner)
end

local function _isGroupAlive(grp)
  if grp == nil then return false end

  -- MOOSE GROUP wrapper
  if grp.IsAlive then
    local ok, alive = _safeCall(grp.IsAlive, grp)
    return ok and alive == true
  end

  -- DCS Group object
  if grp.isExist then
    local okE, exists = _safeCall(grp.isExist, grp)
    if not okE or not exists then return false end
  end
  if grp.getUnits then
    local okU, units = _safeCall(grp.getUnits, grp)
    if okU and type(units) == "table" then
      for _, unit in ipairs(units) do
        if unit and unit.isExist and unit.getLife then
          local okUE, uexists = _safeCall(unit.isExist, unit)
          if okUE and uexists then
            local activeOk = true
            local active = true

            -- Late-activated template units may "exist" but not be active.
            if unit.isActive then
              activeOk, active = _safeCall(unit.isActive, unit)
            end

            if (not activeOk) or active then
              local okL, life = _safeCall(unit.getLife, unit)
              if okL and (life or 0) > 0 then
                return true
              end
            end
          end
        end
      end
    end
  end

  return false
end

local function _findAliveDCSGroupByPrefix(coalitionSide, groupCategory, prefix)
  if not (coalition and coalition.getGroups) then return nil end
  if type(prefix) ~= "string" or prefix == "" then return nil end

  local ok, groups = _safeCall(coalition.getGroups, coalitionSide, groupCategory)
  if not ok or type(groups) ~= "table" then return nil end

  for _, grp in ipairs(groups) do
    if grp and grp.getName then
      local okN, name = _safeCall(grp.getName, grp)
      if okN and type(name) == "string" and _startsWith(name, prefix) then
        if _isGroupAlive(grp) then
          return grp, name
        end
      end
    end
  end

  return nil
end

local function _nowSeconds()
  return (timer and timer.getTime) and timer.getTime() or 0
end

local function _trim(s)
  if type(s) ~= "string" then return s end
  return (s:gsub("^%s+", ""):gsub("%s+$", ""))
end

local function _dist2D(a, b)
  if not a or not b then return nil end
  local dx = (a.x or 0) - (b.x or 0)
  local dy = (a.y or 0) - (b.y or 0)
  return math.sqrt(dx * dx + dy * dy)
end

local function _toVec2FromPoint(point)
  if not point then return nil end
  return { x = point.x, y = point.z }
end

local function _explodePoint(point, power)
  if not (trigger and trigger.action and trigger.action.explosion) then
    return
  end
  if not point then return end
  _safeCall(trigger.action.explosion, point, power or 100)
end

local function _normalizeRelPath(rel)
  if type(rel) ~= "string" then return nil end
  -- DCS `io.open` accepts both, but keep it consistent.
  rel = rel:gsub("\\", "/")
  rel = rel:gsub("//+", "/")
  return rel
end

local function _ensureDir(path)
  if not path or not lfs or not lfs.mkdir then return end
  _safeCall(lfs.mkdir, path)
end

local function _dirname(path)
  if type(path) ~= "string" then return nil end
  local normalized = path:gsub("\\", "/")
  local dir = normalized:match("^(.*)/[^/]*$")
  return dir
end

local function _isFileNonEmpty(path)
  if type(path) ~= "string" then return false end
  local f = io and io.open and io.open(path, "r")
  if not f then return false end
  local ok, content = _safeCall(function() return f:read("*a") end)
  _safeCall(function() f:close() end)
  if not ok or type(content) ~= "string" then return false end
  return _trim(content) ~= ""
end

WW2PacCVOps.Config = {
  Airfields = {
    "Orote",
    "Agana",
    "Gurguan Point",
    "Ushi",
    "Airfield 3",
    "Isley",
    "Charon Kanoa",
    "Kagman",
    "Marpi",
  },

  -- Radius around the airbase used to decide if red ground units remain.
  -- Must encompass the typical airfield defenses.
  ClearRadiusMeters = 2100,

  -- Periodic checks (seconds)
  AirfieldTickSeconds = 20,
  AirfieldTickInitialDelay = 5,
  RespawnTickSeconds = 30,

  -- Landing artillery zones: "Landing Arty Zone-X"
  LandingArtyMinDelaySeconds = 1,
  LandingArtyMaxDelaySeconds = 8,
  LandingArtyExplosionPower = 8,

  -- Landing infantry templates: "Landing-X-Inf-Y"
  LandingInfantryRespawnTickSeconds = 5,

  -- Rando landing templates: "Rando-Landing-X"
  RandoLandingRespawnTickSeconds = 5,
  RandoLandingRespawnDelaySeconds = 10 * 60,

  -- Mission start grace: delay CAP spawns/respawns and suppress early neutralization checks
  -- to allow late-activated airfield ground units to spawn in.
  InitialCAPDelaySeconds = 60,
  InitialRedfieldGraceSeconds = 60,

  -- Extra logging to help troubleshoot AI spawns.
  DebugCAP = true,

  -- Production safety: disable the TEST F10 menu by default.
  EnableTestMenu = false,

  -- Victory
  VictoryPicture = "victory-bluefor.png",
  VictoryPictureSeconds = 5,

  -- Some DCS environments do not expose trigger.action.endMission/outPicture to scripting.
  -- As a fallback, the script will set this user flag to 1 on victory so the Mission Editor
  -- can end the mission via a trigger.
  VictoryEndUserFlag = 9001,

  -- Persistence
  -- Stored under Saved Games (lfs.writedir()) so it survives mission restarts.
  -- User requested path: Missions\\WWIIPacCVOps\\persisted-state.dat
  PersistenceRelativePath = "Missions\\WWIIPacCVOps\\persisted-state.dat",

  -- Hygiene restart
  -- After this many seconds from mission start, persist state and end mission
  -- (no victor) so DCS can restart the mission and restore from persistence.
  HygieneRestartSeconds = 4 * 60 * 60,
  HygieneRestartWarningMinutes = { 30, 15, 5 },
}

WW2PacCVOps.State = {
  Fields = {},
  VictoryTriggered = false,
  Convoys = {},
  A2G = {},
  StartTime = 0,
  CAPEnabled = false,
  HygieneRestartTriggered = false,

  LandingArty = {
    Zones = {},
    Enabled = false,
  },

  LandingInfantry = {
    Templates = {},
  },

  RandoLanding = {
    Templates = {},
  },
}

local function _clamp(n, minV, maxV)
  n = tonumber(n)
  minV = tonumber(minV)
  maxV = tonumber(maxV)
  if not n or not minV or not maxV then return n end
  if n < minV then return minV end
  if n > maxV then return maxV end
  return n
end

-- DCS may sandbox away math.randomseed. Provide a safe PRNG for our own timing needs.
local function _randIntInclusive(stateTable, minV, maxV)
  minV = tonumber(minV)
  maxV = tonumber(maxV)
  if not minV or not maxV then return minV end
  if maxV < minV then maxV = minV end

  if math and math.random then
    -- math.randomseed may not exist; but math.random typically does.
    return math.random(minV, maxV)
  end

  -- LCG fallback (Numerical Recipes constants)
  local seed = tonumber(stateTable._RngSeed) or 123456789
  seed = (1664525 * seed + 1013904223) % 4294967296
  stateTable._RngSeed = seed
  local span = (maxV - minV + 1)
  if span <= 1 then return minV end
  return minV + (seed % span)
end

local function _isDcsTimerAvailable()
  return timer and timer.getTime and timer.scheduleFunction
end

-- Forward declaration: landing infantry tick uses this before its definition.
local _groupUnitCount

local function _getGroundHeightAt(vec3)
  if not (land and land.getHeight) then return nil end
  if not vec3 or type(vec3.x) ~= "number" then return nil end
  -- DCS land.getHeight expects Vec2-like: { x = <east>, y = <north> } where y is vec3.z.
  local ok, h = _safeCall(land.getHeight, { x = vec3.x, y = vec3.z })
  if ok then return h end
  return nil
end

function WW2PacCVOps:_initLandingArtyZones()
  if not _isDcsTimerAvailable() then
    _warn("DCS timer not available; landing artillery zones disabled")
    return
  end

  if not (SET_ZONE and SET_ZONE.New) then
    _warn("MOOSE SET_ZONE not available; landing artillery zones disabled")
    return
  end

  local minDelay = _clamp(self.Config.LandingArtyMinDelaySeconds, 0.1, 3600) or 1
  local maxDelay = _clamp(self.Config.LandingArtyMaxDelaySeconds, minDelay, 3600) or 3
  local power = tonumber(self.Config.LandingArtyExplosionPower) or 15

  -- Seed fallback RNG (used only if math.random is not available).
  if not self.State._RngSeed then
    local seed = 0
    if timer and timer.getAbsTime then
      local ok, t = _safeCall(timer.getAbsTime)
      if ok and t then seed = t end
    elseif timer and timer.getTime then
      local ok, t = _safeCall(timer.getTime)
      if ok and t then seed = t end
    end
    self.State._RngSeed = math.floor((tonumber(seed) or 0) * 1000) + 1
  end

  local zoneSet = SET_ZONE:New():FilterPrefixes("Landing Arty Zone-"):FilterOnce()
  local found = 0

  local function startZone(zone)
    if not zone then return end
    local name = zone.GetName and zone:GetName() or "<unknown zone>"
    self.State.LandingArty.Zones[name] = zone
    found = found + 1

    local function scheduleNext()
      local delay = minDelay
      if maxDelay > minDelay then
        delay = _randIntInclusive(self.State, minDelay, maxDelay)
      end

      timer.scheduleFunction(function()
        -- If mission is ending/restarting, stop scheduling new rounds.
        if self.State.VictoryTriggered or self.State.HygieneRestartTriggered then
          return nil
        end

        if trigger and trigger.action and trigger.action.explosion then
          local vec3 = nil
          if zone.GetRandomPointVec3 then
            local ok, p = _safeCall(zone.GetRandomPointVec3, zone)
            if ok then vec3 = p end
          end
          if vec3 and type(vec3) == "table" and type(vec3.x) == "number" then
            local h = _getGroundHeightAt(vec3)
            if type(h) == "number" then
              vec3.y = h
            end
            _safeCall(trigger.action.explosion, vec3, power)
          end
        end

        scheduleNext()
        return nil
      end, nil, timer.getTime() + delay)
    end

    scheduleNext()
    _log("Landing arty active: " .. tostring(name))
  end

  if zoneSet and zoneSet.ForEachZone then
    zoneSet:ForEachZone(function(z)
      startZone(z)
    end)
  end

  if found > 0 then
    self.State.LandingArty.Enabled = true
    _log("Landing arty zones initialized: " .. tostring(found))
  else
    _log("Landing arty zones: none found (expected names like 'Landing Arty Zone-1')")
  end
end

function WW2PacCVOps:_initLandingInfantryRespawns()
  if not (SET_GROUP and SET_GROUP.New) then
    _warn("MOOSE SET_GROUP not available; landing infantry respawn disabled")
    return
  end

  local infantrySet = SET_GROUP:New()
    :FilterPrefixes("Landing-")
    :FilterFunction(function(grp)
      if not grp or not grp.GetName then return false end
      local name = grp:GetName()
      if type(name) ~= "string" then return false end
      return name:match("^Landing%-%d+%-Inf%-%d+$") ~= nil
    end)
    :FilterOnce()

  local templates = {}
  local count = 0

  infantrySet:ForEachGroup(function(grp)
    local name = grp:GetName()
    count = count + 1
    local sp = _makeSpawner(name)
    templates[name] = {
      Template = name,
      Spawner = sp,
      Group = nil,
      SpawnFailures = 0,
      NextSpawnTime = 0,
    }
  end)

  self.State.LandingInfantry.Templates = templates

  if count <= 0 then
    _log("Landing infantry: none found (expected names like 'Landing-3-Inf-10')")
    return
  end

  _log("Landing infantry templates initialized: " .. tostring(count))

  -- Spawn one instance of each template at mission start.
  for _, t in pairs(self.State.LandingInfantry.Templates) do
    if t.Spawner then
      t.Group = _safeSpawn(t.Spawner)
    end
  end
end

function WW2PacCVOps:_tickLandingInfantry()
  if type(self.State.LandingInfantry.Templates) ~= "table" then return end

  local now = _nowSeconds()

  for _, t in pairs(self.State.LandingInfantry.Templates) do
    if t.Spawner and not _isGroupAlive(t.Group) then
      local nextAllowed = t.NextSpawnTime or 0
      if now >= nextAllowed then
        t.Group = _safeSpawn(t.Spawner)

        local alive = _isGroupAlive(t.Group)
        local units = _groupUnitCount(t.Group) or 0
        if alive and units > 0 then
          t.SpawnFailures = 0
          t.NextSpawnTime = 0
        else
          local failures = (t.SpawnFailures or 0) + 1
          t.SpawnFailures = failures

          -- Backoff capped at 2 minutes to avoid log spam if a template is broken.
          local delay = math.min(120, 5 * math.pow(2, math.min(failures - 1, 5)))
          t.NextSpawnTime = now + delay
        end
      end
    end
  end
end

function WW2PacCVOps:_initRandoLandingRespawns()
  if not (SET_GROUP and SET_GROUP.New) then
    _warn("MOOSE SET_GROUP not available; rando landing respawn disabled")
    return
  end

  local randoSet = SET_GROUP:New()
    :FilterPrefixes("Rando-Landing-")
    :FilterFunction(function(grp)
      if not grp or not grp.GetName then return false end
      local name = grp:GetName()
      if type(name) ~= "string" then return false end
      return name:match("^Rando%-%Landing%-%d+$") ~= nil
    end)
    :FilterOnce()

  local templates = {}
  local count = 0

  randoSet:ForEachGroup(function(grp)
    local name = grp:GetName()
    count = count + 1
    local sp = _makeSpawner(name)
    templates[name] = {
      Template = name,
      Spawner = sp,
      Group = nil,
      DeadAt = nil,
      RespawnAt = nil,
      SpawnFailures = 0,
      NextSpawnTime = 0,
    }
  end)

  self.State.RandoLanding.Templates = templates

  if count <= 0 then
    _log("Rando landing: none found (expected names like 'Rando-Landing-1')")
    return
  end

  _log("Rando landing templates initialized: " .. tostring(count))

  -- Spawn one instance of each template at mission start.
  for _, t in pairs(self.State.RandoLanding.Templates) do
    if t.Spawner then
      t.Group = _safeSpawn(t.Spawner)
    end
  end
end

function WW2PacCVOps:_tickRandoLanding()
  if type(self.State.RandoLanding.Templates) ~= "table" then return end

  local now = _nowSeconds()
  local respawnDelay = tonumber(self.Config.RandoLandingRespawnDelaySeconds) or (10 * 60)

  for _, t in pairs(self.State.RandoLanding.Templates) do
    if t.Spawner then
      if _isGroupAlive(t.Group) then
        -- Reset timers when alive.
        t.DeadAt = nil
        t.RespawnAt = nil
      else
        -- Mark first observed death.
        if not t.DeadAt then
          t.DeadAt = now
          t.RespawnAt = now + respawnDelay
        end

        -- Respect any spawn backoff if a template is broken.
        local nextAllowed = t.NextSpawnTime or 0
        if t.RespawnAt and now >= t.RespawnAt and now >= nextAllowed then
          t.Group = _safeSpawn(t.Spawner)

          local alive = _isGroupAlive(t.Group)
          local units = _groupUnitCount(t.Group) or 0
          if alive and units > 0 then
            t.SpawnFailures = 0
            t.NextSpawnTime = 0
            t.DeadAt = nil
            t.RespawnAt = nil
          else
            local failures = (t.SpawnFailures or 0) + 1
            t.SpawnFailures = failures

            -- Retry with mild backoff (caps at 2 minutes), but do not reset RespawnAt.
            local delay = math.min(120, 10 * math.pow(2, math.min(failures - 1, 4)))
            t.NextSpawnTime = now + delay
          end
        end
      end
    end
  end
end

function WW2PacCVOps:_broadcastText(msg, seconds)
  if not msg then return end
  seconds = seconds or 15

  if trigger and trigger.action and trigger.action.outText then
    _safeCall(trigger.action.outText, tostring(msg), seconds)
    return
  end

  _warn("trigger.action.outText not available")
end

function WW2PacCVOps:_broadcastTextForCoalition(side, msg, seconds)
  if not msg then return end
  seconds = seconds or 15

  if trigger and trigger.action and trigger.action.outTextForCoalition then
    _safeCall(trigger.action.outTextForCoalition, side, tostring(msg), seconds)
    return
  end

  -- Fallback: broadcast to all.
  self:_broadcastText(msg, seconds)
end

function WW2PacCVOps:_setFieldCircle(field, side)
  if not field or not field.Zone or not field.Zone.DrawZone then
    return
  end

  local lineRgb = { 0, 0, 0 }
  local lineAlpha = 1
  local fillRgb = { 1, 0, 0 }
  local fillAlpha = 0.2

  -- Desired styling:
  -- RED:    red fill + black line
  -- NEUTRAL: 50% gray fill + green line
  -- BLUE:   50% blue fill + black line
  if side == coalition.side.BLUE then
    lineRgb = { 0, 0, 0 }
    fillRgb = { 0, 0, 1 }
    fillAlpha = 0.2
  elseif side == coalition.side.NEUTRAL then
    lineRgb = { 0, 1, 0 }
    fillRgb = { 0.5, 0.5, 0.5 }
    fillAlpha = 0.2
  else
    -- RED (default)
    lineRgb = { 0, 0, 0 }
    fillRgb = { 1, 0, 0 }
    fillAlpha = 0.2
  end

  if field.Zone.UndrawZone then
    field.Zone:UndrawZone()
  end

  if field.Zone.SetColor then
    field.Zone:SetColor(lineRgb, lineAlpha)
  end
  if field.Zone.SetFillColor then
    field.Zone:SetFillColor(fillRgb, fillAlpha)
  end

  -- Coalition -1 draws to all.
  field.Zone:DrawZone(-1, lineRgb, lineAlpha, fillRgb, fillAlpha, 1, true)
end

function WW2PacCVOps:_drawInitialFieldCircles()
  for _, field in pairs(self.State.Fields) do
    if field.State == "RED" then
      self:_setFieldCircle(field, coalition.side.RED)
    end
  end
end

_groupUnitCount = function(grp)
  if not grp then return nil end
  if grp.CountAliveUnits then
    local ok, n = _safeCall(grp.CountAliveUnits, grp)
    if ok then return n end
  end
  if grp.GetUnits then
    local ok, units = _safeCall(grp.GetUnits, grp)
    if ok and type(units) == "table" then
      return #units
    end
  end
  return nil
end

function WW2PacCVOps:_debugLogGroup(label, grp)
  if not self.Config.DebugCAP then return end

  if not grp then
    _warn(label .. ": spawn returned nil")
    return
  end

  local name = (grp.GetName and grp:GetName()) or "<unknown>"
  local alive = _isGroupAlive(grp)
  local units = _groupUnitCount(grp)
  local pos = nil
  if grp.GetCoordinate then
    local ok, coord = _safeCall(grp.GetCoordinate, grp)
    if ok and coord and coord.ToStringLLDDM then
      pos = coord:ToStringLLDDM()
    end
  end

  _log(string.format("%s: %s alive=%s units=%s pos=%s", label, tostring(name), tostring(alive), tostring(units), tostring(pos)))
end

function WW2PacCVOps:_initFields()
  for _, name in ipairs(self.Config.Airfields) do
    local ab = AIRBASE and AIRBASE.FindByName and AIRBASE:FindByName(name) or nil
    if not ab then
      _warn("AIRBASE not found: '" .. tostring(name) .. "' (check airbase name in mission)")
    end

    local zone = nil
    if ab and ZONE_RADIUS and ZONE_RADIUS.New then
      local ok, vec2 = _safeCall(ab.GetVec2, ab)
      if ok and vec2 then
        zone = ZONE_RADIUS:New("WW2PacCVOps-" .. name, vec2, self.Config.ClearRadiusMeters)
      end
    end

    local field = {
      Name = name,
      Airbase = ab,
      Zone = zone,
      State = "RED", -- RED -> NEUTRAL -> BLUE
      CaptureLanded = false,
      Captured = false,

      CaptureSpawner = _makeSpawner(name .. "-Capture"),
      OccupationSpawner = _makeSpawner(name .. "-Occupation"),

      CAPSpawners = {
        _makeSpawner(name .. "-CAP-1"),
        _makeSpawner(name .. "-CAP-2"),
      },

      CAPGroups = { nil, nil },
      CAPSpawnFailures = { 0, 0 },
      CAPNextSpawnTime = { 0, 0 },
      CaptureGroup = nil,

      CaptureSpawnFailures = 0,
      CaptureNextSpawnTime = 0,
    }

    self.State.Fields[name] = field
  end
end

function WW2PacCVOps:_spawnInitialCAP()
  for _, field in pairs(self.State.Fields) do
    -- With persistence, some fields may already be BLUE at mission start.
    -- Only spawn CAP for RED fields.
    if field.State == "RED" then
      for i = 1, 2 do
        if field.CAPSpawners[i] then
          field.CAPNextSpawnTime[i] = 0
          field.CAPSpawnFailures[i] = 0
          field.CAPGroups[i] = _safeSpawn(field.CAPSpawners[i])
          self:_debugLogGroup(string.format("Initial CAP-%d spawn for %s", i, field.Name), field.CAPGroups[i])

          if not _isGroupAlive(field.CAPGroups[i]) or (_groupUnitCount(field.CAPGroups[i]) or 0) <= 0 then
            field.CAPSpawnFailures[i] = 1
            field.CAPNextSpawnTime[i] = _nowSeconds() + 60
            _warn(string.format(
              "CAP-%d for %s spawned dead/empty; delaying retries. Common causes: aircraft type/module not installed/invalid, group set to Client/Player, or template misconfigured.",
              i, field.Name
            ))
          end
        end
      end
    end
  end
end

function WW2PacCVOps:_getPersistencePath()
  local rel = _normalizeRelPath(self.Config.PersistenceRelativePath)
  if not rel then return nil end

  local base = nil
  if lfs and lfs.writedir then
    local ok, dir = _safeCall(lfs.writedir)
    if ok then base = dir end
  end
  if not base then base = "./" end

  base = base:gsub("\\", "/")
  if not base:match("/$") then
    base = base .. "/"
  end

  local full = base .. rel
  full = full:gsub("//+", "/")
  return full
end

function WW2PacCVOps:_persistReadBlueFields()
  local path = self:_getPersistencePath()
  if not path then return {} end
  if not _isFileNonEmpty(path) then return {} end

  local f = io and io.open and io.open(path, "r")
  if not f then
    _warn("Persistence file exists but could not be opened for read: " .. tostring(path))
    return {}
  end

  local fields = {}
  local seen = {}
  for line in f:lines() do
    local name = _trim(line)
    if name ~= "" and not seen[name] then
      table.insert(fields, name)
      seen[name] = true
    end
  end
  _safeCall(function() f:close() end)
  return fields
end

function WW2PacCVOps:_persistWriteBlueFields(blueFieldNames)
  local path = self:_getPersistencePath()
  if not path then
    _warn("Persistence path is nil; cannot write persisted state")
    return
  end

  local dir = _dirname(path)
  if dir then
    -- Ensure Missions/ and Missions/WWIIPacCVOps exist under writedir.
    local parts = {}
    for part in dir:gsub("\\", "/"):gsub("//+", "/"):gmatch("[^/]+") do
      table.insert(parts, part)
    end
    if #parts > 0 then
      -- Rebuild incrementally to create nested directories.
      local prefix = dir:match("^/" ) and "/" or ""
      local accum = prefix
      for i = 1, #parts do
        if accum ~= "" and not accum:match("/$") then accum = accum .. "/" end
        accum = accum .. parts[i]
        _ensureDir(accum)
      end
    end
  end

  local f = io and io.open and io.open(path, "w")
  if not f then
    _warn("Could not open persistence file for write: " .. tostring(path))
    return
  end

  if type(blueFieldNames) == "table" then
    for _, name in ipairs(blueFieldNames) do
      if type(name) == "string" and name ~= "" then
        f:write(name)
        f:write("\n")
      end
    end
  end

  _safeCall(function() f:close() end)
end

function WW2PacCVOps:_persistWriteCurrentBlueFields()
  local names = {}
  for _, field in pairs(self.State.Fields) do
    if field and field.Captured == true then
      table.insert(names, field.Name)
    end
  end
  table.sort(names)
  self:_persistWriteBlueFields(names)
end

function WW2PacCVOps:_persistClear()
  -- Clear to empty file so next cycle starts from default (all Red).
  self:_persistWriteBlueFields({})
end

function WW2PacCVOps:_removeRedGroundInFieldZone(field)
  if not field or not field.Zone then return end

  field.Zone:Scan({ Object.Category.UNIT }, { Unit.Category.GROUND_UNIT })
  local redSet = field.Zone:GetScannedSetGroup(coalition.side.RED)
  if not redSet then return end

  if redSet.ForEachGroupAlive then
    redSet:ForEachGroupAlive(function(grp)
      if grp and grp.Destroy then
        _safeCall(grp.Destroy, grp)
      end
    end)
  end
end

function WW2PacCVOps:_applyPersistedBlueField(field)
  if not field or field.Captured then return end

  -- Ensure any remaining Red ground defenders are removed.
  self:_removeRedGroundInFieldZone(field)

  field.Captured = true
  field.CaptureLanded = true
  field.State = "BLUE"

  if field.Airbase then
    _setAirbaseCoalition(field.Airbase, coalition.side.BLUE)
  end

  if field.OccupationSpawner then
    _safeSpawn(field.OccupationSpawner)
  end

  self:_setFieldCircle(field, coalition.side.BLUE)
end

function WW2PacCVOps:_applyPersistenceOnStart()
  local path = self:_getPersistencePath()
  if not path then
    _warn("Persistence path is nil; skipping persistence")
    return
  end

  local persisted = self:_persistReadBlueFields()
  if #persisted <= 0 then
    _log("Persistence: no saved BLUE airfields (" .. tostring(path) .. ")")
    return
  end

  _log("Persistence: applying " .. tostring(#persisted) .. " BLUE airfields from " .. tostring(path))
  for _, name in ipairs(persisted) do
    local field = self.State.Fields[name]
    if field then
      self:_applyPersistedBlueField(field)
    else
      _warn("Persistence: unknown airfield name in file: '" .. tostring(name) .. "'")
    end
  end
end

function WW2PacCVOps:_scheduleHygieneRestart()
  if not (timer and timer.scheduleFunction and timer.getTime) then
    _warn("timer.scheduleFunction not available; hygiene restart disabled")
    return
  end

  local restartAfter = tonumber(self.Config.HygieneRestartSeconds) or (4 * 60 * 60)
  if restartAfter <= 0 then
    _warn("HygieneRestartSeconds <= 0; hygiene restart disabled")
    return
  end

  local startTime = self.State.StartTime or timer.getTime()
  local restartAt = startTime + restartAfter

  local warnings = self.Config.HygieneRestartWarningMinutes
  if type(warnings) == "table" then
    for _, minutes in ipairs(warnings) do
      local m = tonumber(minutes)
      if m and m > 0 then
        local warnAt = restartAt - (m * 60)
        if warnAt > timer.getTime() then
          timer.scheduleFunction(function()
            if self.State.VictoryTriggered or self.State.HygieneRestartTriggered then
              return nil
            end
            local msg = "Attention Aviators, the server will perform a restart to clear memory.  Map state will persist.  Restart in " .. tostring(m) .. " minutes."
            self:_broadcastText(msg, 15)
            return nil
          end, nil, warnAt)
        end
      end
    end
  end

  timer.scheduleFunction(function()
    if self.State.VictoryTriggered or self.State.HygieneRestartTriggered then
      return nil
    end

    self.State.HygieneRestartTriggered = true

    -- Persist the latest state right before restart.
    self:_persistWriteCurrentBlueFields()

    self:_broadcastText(
      "Attention Aviators, the server will now perform a restart to clear memory.  Map state will persist.",
      15
    )

    if trigger and trigger.action and trigger.action.endMission then
      _safeCall(trigger.action.endMission)
    else
      _warn("trigger.action.endMission not available")
    end

    return nil
  end, nil, restartAt)

  _log(string.format("Hygiene restart scheduled in %ss (at t=%.1f)", tostring(restartAfter), restartAt))
end

function WW2PacCVOps:_countRedGroundInZone(field)
  if not field.Zone then
    return nil
  end

  -- Scan for ground UNITs only.
  field.Zone:Scan({ Object.Category.UNIT }, { Unit.Category.GROUND_UNIT })

  local redSet = field.Zone:GetScannedSetGroup(coalition.side.RED)
  if not redSet then
    return 0
  end

  local _, unitCount = redSet:CountAlive()
  return unitCount or 0
end

function WW2PacCVOps:_neutralizeField(field)
  if field.State ~= "RED" then return end

  field.State = "NEUTRAL"
  _log("Airfield neutralized: " .. field.Name)

  if field.Airbase then
    _setAirbaseCoalition(field.Airbase, coalition.side.NEUTRAL)
  end

  -- Update circle styling for NEUTRAL.
  self:_setFieldCircle(field, coalition.side.NEUTRAL)

  -- Stop CAP respawns by virtue of State check.
  -- Spawn capture C-47 now.
  self:_ensureCaptureAircraft(field)
end

function WW2PacCVOps:_captureField(field)
  if field.Captured then return end

  field.Captured = true
  field.State = "BLUE"

  _log("Airfield captured for BLUE: " .. field.Name)

  if field.Airbase then
    _setAirbaseCoalition(field.Airbase, coalition.side.BLUE)
  end

  if field.OccupationSpawner then
    _safeSpawn(field.OccupationSpawner)
  end

  -- Replace the red circle with a blue one.
  self:_setFieldCircle(field, coalition.side.BLUE)

  local function countRemaining()
    local n = 0
    for _, f in pairs(self.State.Fields) do
      if not f.Captured then
        n = n + 1
      end
    end
    return n
  end

  local numRedforAirfields = countRemaining()
  local msg = "Attention Aviators, our forces have captured another airbase.  There are " .. tostring(numRedforAirfields) .. " Redfor Airfields left to capture."
  if trigger and trigger.action and trigger.action.outTextForCoalition then
    _safeCall(trigger.action.outTextForCoalition, coalition.side.BLUE, msg, 15)
  elseif trigger and trigger.action and trigger.action.outText then
    _safeCall(trigger.action.outText, msg, 15)
  else
    _warn("trigger.action.outTextForCoalition not available")
  end

  self:_checkVictory()

  -- Persist newly captured BLUE airfield.
  self:_persistWriteCurrentBlueFields()
end

function WW2PacCVOps:_ensureCaptureAircraft(field)
  if field.State ~= "NEUTRAL" or field.CaptureLanded or field.Captured then
    return
  end

  if _isGroupAlive(field.CaptureGroup) then
    return
  end

  -- If our stored handle went stale, try to detect an already-spawned capture aircraft by name.
  local prefix = tostring(field.Name) .. "-Capture"
  local existingDcsGroup, existingName = _findAliveDCSGroupByPrefix(coalition.side.BLUE, Group.Category.AIRPLANE, prefix)
  if existingDcsGroup then
    return
  end

  local now = _nowSeconds()
  local nextAllowed = field.CaptureNextSpawnTime or 0
  if now < nextAllowed then
    return
  end

  if not field.CaptureSpawner then
    _warn("Capture spawner missing for " .. tostring(field.Name) .. ". Check template group '" .. tostring(field.Name) .. "-Capture' exists and is late-activated AI.")
    -- Backoff to avoid spamming.
    field.CaptureNextSpawnTime = now + 60
    return
  end

  field.CaptureGroup = _safeSpawn(field.CaptureSpawner)

  local alive = _isGroupAlive(field.CaptureGroup)
  local units = _groupUnitCount(field.CaptureGroup) or 0
  if field.CaptureGroup and alive and units > 0 then
    field.CaptureSpawnFailures = 0
    field.CaptureNextSpawnTime = 0

    _log("Spawned capture C-47 for " .. field.Name)
    local msg = "C-47 with occupation troops are inbound to capture airfield: " .. tostring(field.Name) .. ".  Protect it from attack by patrols."
    self:_broadcastTextForCoalition(coalition.side.BLUE, msg, 15)
    return
  end
  field.CaptureSpawnFailures = (field.CaptureSpawnFailures or 0) + 1
  local failures = field.CaptureSpawnFailures
  local delay = math.min(600, 30 * math.pow(2, math.min(failures - 1, 4)))
  field.CaptureNextSpawnTime = now + delay

  _warn(string.format(
    "Capture aircraft for %s spawned dead/empty (failures=%d). Next retry in %ds. Check template '%s-Capture' (AI, late-activated, valid aircraft type/module).",
    tostring(field.Name), failures, delay, tostring(field.Name)
  ))
end

function WW2PacCVOps:_ensureCAP(field)
  if not self.State.CAPEnabled then
    return
  end

  if field.State ~= "RED" then
    return
  end

  for i = 1, 2 do
    if field.CAPSpawners[i] and not _isGroupAlive(field.CAPGroups[i]) then
      local now = _nowSeconds()
      local nextAllowed = field.CAPNextSpawnTime[i] or 0

      if now >= nextAllowed then
        field.CAPGroups[i] = _safeSpawn(field.CAPSpawners[i])
        _log(string.format("Respawned CAP-%d for %s", i, field.Name))
        self:_debugLogGroup(string.format("Respawn CAP-%d for %s", i, field.Name), field.CAPGroups[i])

        local alive = _isGroupAlive(field.CAPGroups[i])
        local units = _groupUnitCount(field.CAPGroups[i]) or 0
        if alive and units > 0 then
          field.CAPSpawnFailures[i] = 0
          field.CAPNextSpawnTime[i] = 0
        else
          local failures = (field.CAPSpawnFailures[i] or 0) + 1
          field.CAPSpawnFailures[i] = failures

          -- Exponential-ish backoff capped at 10 minutes.
          local delay = math.min(600, 30 * math.pow(2, math.min(failures - 1, 4)))
          field.CAPNextSpawnTime[i] = now + delay

          _warn(string.format(
            "CAP-%d for %s spawned dead/empty (failures=%d). Next retry in %ds. Check unit type in template '%s'.",
            i, field.Name, failures, delay, tostring(field.Name .. "-CAP-" .. i)
          ))
        end
      end
    end
  end
end

function WW2PacCVOps:_tickAirfields()
  for _, field in pairs(self.State.Fields) do
    if field.State == "RED" then
      self:_ensureCAP(field)

      local now = _nowSeconds()
      local graceUntil = (self.State.StartTime or 0) + (self.Config.InitialRedfieldGraceSeconds or 0)
      if now >= graceUntil then
        local redUnits = self:_countRedGroundInZone(field)
        if redUnits ~= nil and redUnits <= 0 then
          self:_neutralizeField(field)
        end
      end

    elseif field.State == "NEUTRAL" then
      self:_ensureCaptureAircraft(field)
    end
  end
end

function WW2PacCVOps:_checkVictory()
  if self.State.VictoryTriggered then return end

  for _, field in pairs(self.State.Fields) do
    if not field.Captured then
      return
    end
  end

  self.State.VictoryTriggered = true

  _log("All airfields captured. Triggering BLUE victory.")

  -- Mission Editor will handle image + mission end.
  -- DCS user flags are numeric; use 1 as "true".
  if trigger and trigger.action and trigger.action.setUserFlag then
    _safeCall(trigger.action.setUserFlag, "victory", 1)
    _log("Victory: set user flag 'victory'=1")
  else
    _warn("trigger.action.setUserFlag not available; cannot signal ME victory")
  end

  -- Victory resets the campaign cycle: clear persistence so next start is all RED.
  self:_persistClear()
end

function WW2PacCVOps:_initConvoys()
  local convoyTemplates = {
    "Ground-1",
    "Ground-2",
    "Ground-3",
    "Ground-4",
    "Ground-5",
    "Ground-6",
  }

  for _, tpl in ipairs(convoyTemplates) do
    local sp = _makeSpawner(tpl)
    self.State.Convoys[tpl] = {
      Template = tpl,
      Spawner = sp,
      Group = nil,
    }
  end
end

function WW2PacCVOps:_initA2G()
  local a2gTemplates = {
    "A2G-1",
    "A2G-2",
    "A2G-3",
    "A2G-4",
  }

  for _, tpl in ipairs(a2gTemplates) do
    local sp = _makeSpawner(tpl)
    self.State.A2G[tpl] = {
      Template = tpl,
      Spawner = sp,
      Group = nil,
      SpawnFailures = 0,
      NextSpawnTime = 0,
    }
  end
end

function WW2PacCVOps:_spawnInitialConvoys()
  for _, c in pairs(self.State.Convoys) do
    if c.Spawner then
      c.Group = _safeSpawn(c.Spawner)
    end
  end
end

function WW2PacCVOps:_spawnInitialA2G()
  for _, a in pairs(self.State.A2G) do
    if a.Spawner then
      a.NextSpawnTime = 0
      a.SpawnFailures = 0

      a.Group = _safeSpawn(a.Spawner)
      _log("Spawned initial A2G: " .. tostring(a.Template))
      self:_debugLogGroup("Initial A2G spawn: " .. tostring(a.Template), a.Group)

      local alive = _isGroupAlive(a.Group)
      local units = _groupUnitCount(a.Group) or 0
      if (not alive) or units <= 0 then
        a.SpawnFailures = 1
        a.NextSpawnTime = _nowSeconds() + 60
        _warn(string.format(
          "A2G %s spawned dead/empty; delaying retries. Common causes: aircraft type/module not installed/invalid, group set to Client/Player, or template misconfigured.",
          tostring(a.Template)
        ))
      end
    end
  end
end

function WW2PacCVOps:_tickConvoys()
  for _, c in pairs(self.State.Convoys) do
    if c.Spawner and not _isGroupAlive(c.Group) then
      c.Group = _safeSpawn(c.Spawner)
      if c.Group then
        _log("Respawned convoy: " .. c.Template)
      end
    end
  end
end

function WW2PacCVOps:_tickA2G()
  for _, a in pairs(self.State.A2G) do
    if a.Spawner and not _isGroupAlive(a.Group) then
      local now = _nowSeconds()
      local nextAllowed = a.NextSpawnTime or 0

      if now >= nextAllowed then
        a.Group = _safeSpawn(a.Spawner)
        _log("Respawned A2G: " .. tostring(a.Template))
        self:_debugLogGroup("Respawn A2G: " .. tostring(a.Template), a.Group)

        local alive = _isGroupAlive(a.Group)
        local units = _groupUnitCount(a.Group) or 0
        if alive and units > 0 then
          a.SpawnFailures = 0
          a.NextSpawnTime = 0
        else
          local failures = (a.SpawnFailures or 0) + 1
          a.SpawnFailures = failures

          -- Exponential-ish backoff capped at 10 minutes.
          local delay = math.min(600, 30 * math.pow(2, math.min(failures - 1, 4)))
          a.NextSpawnTime = now + delay

          _warn(string.format(
            "A2G %s spawned dead/empty (failures=%d). Next retry in %ds. Check unit type in template '%s'.",
            tostring(a.Template), failures, delay, tostring(a.Template)
          ))
        end
      end
    end
  end
end

function WW2PacCVOps:_initLandingHandler()
  if not EVENTHANDLER or not EVENTS then
    _err("MOOSE EVENTHANDLER/EVENTS not available. Ensure MOOSE is loaded before WW2PacCVOps.lua")
    return
  end

  local handler = EVENTHANDLER:New()
  handler:HandleEvent(EVENTS.Land)

  function handler:OnEventLand(EventData)
    if not EventData or not EventData.IniGroup then
      return
    end

    local groupName = EventData.IniGroup:GetName()
    local landedAt = _getAirbaseNameFromPlace(EventData.Place)

    if not landedAt then
      return
    end

    -- Match any spawned instance like "Agana-Capture#001".
    for _, field in pairs(WW2PacCVOps.State.Fields) do
      local prefix = field.Name .. "-Capture"
      if field.State == "NEUTRAL" and not field.Captured and _startsWith(groupName, prefix) and landedAt == field.Name then
        field.CaptureLanded = true
        _log("Capture aircraft landed at " .. field.Name)
        WW2PacCVOps:_captureField(field)
        return
      end
    end
  end

  self.State.LandHandler = handler
end

function WW2PacCVOps:_initCAPDebugHandlers()
  if not self.Config.DebugCAP then
    return
  end

  if not EVENTHANDLER or not EVENTS then
    return
  end

  local handler = EVENTHANDLER:New()
  handler:HandleEvent(EVENTS.Birth)
  handler:HandleEvent(EVENTS.Dead)
  handler:HandleEvent(EVENTS.Crash)

  function handler:OnEventBirth(EventData)
    if not EventData or not EventData.IniGroup then return end
    local groupName = EventData.IniGroup:GetName()
    if groupName and groupName:find("-CAP-", 1, true) then
      _log("CAP Birth: " .. groupName)
    end
  end

  function handler:OnEventDead(EventData)
    if not EventData or not EventData.IniGroup then return end
    local groupName = EventData.IniGroup:GetName()
    if groupName and groupName:find("-CAP-", 1, true) then
      _log("CAP Dead: " .. groupName)
    end
  end

  function handler:OnEventCrash(EventData)
    if not EventData or not EventData.IniGroup then return end
    local groupName = EventData.IniGroup:GetName()
    if groupName and groupName:find("-CAP-", 1, true) then
      _log("CAP Crash: " .. groupName)
    end
  end

  self.State.CAPDebugHandler = handler
end

function WW2PacCVOps:_getAirfieldCenterVec2(airfieldName)
  local field = self.State.Fields and self.State.Fields[airfieldName] or nil
  if field and field.Airbase and field.Airbase.GetVec2 then
    local ok, vec2 = _safeCall(field.Airbase.GetVec2, field.Airbase)
    if ok and vec2 and vec2.x and vec2.y then
      return { x = vec2.x, y = vec2.y }
    end
  end

  if Airbase and Airbase.getByName then
    local ok, ab = _safeCall(Airbase.getByName, airfieldName)
    if ok and ab and ab.getPoint then
      local ok2, pt = _safeCall(ab.getPoint, ab)
      if ok2 and pt then
        return _toVec2FromPoint(pt)
      end
    end
  end

  return nil
end

function WW2PacCVOps:_explodeDCSGroupUnits(dcsGroup, power)
  if not dcsGroup or not dcsGroup.getUnits then return 0 end
  local ok, units = _safeCall(dcsGroup.getUnits, dcsGroup)
  if not ok or type(units) ~= "table" then return 0 end

  local count = 0
  for _, unit in ipairs(units) do
    if unit and unit.isExist and unit:isExist() and unit.getPoint then
      local okp, pt = _safeCall(unit.getPoint, unit)
      if okp and pt then
        _explodePoint(pt, power)
        count = count + 1
      end
    end
  end
  return count
end

function WW2PacCVOps:_explodeRedGroupsByPrefix(prefix, categories, power)
  if type(prefix) ~= "string" or prefix == "" then return 0 end
  if not (coalition and coalition.getGroups) then return 0 end

  local total = 0
  for _, category in ipairs(categories or {}) do
    local ok, groups = _safeCall(coalition.getGroups, coalition.side.RED, category)
    if ok and type(groups) == "table" then
      for _, grp in ipairs(groups) do
        if grp and grp.getName then
          local okn, name = _safeCall(grp.getName, grp)
          if okn and type(name) == "string" and _startsWith(name, prefix) then
            total = total + self:_explodeDCSGroupUnits(grp, power)
          end
        end
      end
    end
  end
  return total
end

function WW2PacCVOps:_testNukeAgana()
  local center = self:_getAirfieldCenterVec2("Agana")
  if not center then
    _warn("Nuke Agana: could not resolve Agana airbase center")
    return
  end

  local radius = self.Config.ClearRadiusMeters or 2100
  local exploded = 0

  if coalition and coalition.getGroups then
    local ok, groups = _safeCall(coalition.getGroups, coalition.side.RED, Group.Category.GROUND)
    if ok and type(groups) == "table" then
      for _, grp in ipairs(groups) do
        if grp and grp.getUnits then
          local okU, units = _safeCall(grp.getUnits, grp)
          if okU and type(units) == "table" then
            for _, unit in ipairs(units) do
              if unit and unit.isExist and unit:isExist() and unit.getPoint then
                local okP, pt = _safeCall(unit.getPoint, unit)
                if okP and pt then
                  local u2 = _toVec2FromPoint(pt)
                  local d = _dist2D(center, u2)
                  if d and d <= radius then
                    _explodePoint(pt, 120)
                    exploded = exploded + 1
                  end
                end
              end
            end
          end
        end
      end
    end
  end

  if trigger and trigger.action and trigger.action.outText then
    _safeCall(trigger.action.outText, string.format("Nuke Agana: exploded %d red ground units (r=%dm)", exploded, radius), 10)
  end
end

function WW2PacCVOps:_testNukeAganaCAP()
  local categories = { Group.Category.AIRPLANE, Group.Category.HELICOPTER }
  local exploded = 0
  exploded = exploded + self:_explodeRedGroupsByPrefix("Agana-CAP-1", categories, 600)
  exploded = exploded + self:_explodeRedGroupsByPrefix("Agana-CAP-2", categories, 600)

  if trigger and trigger.action and trigger.action.outText then
    _safeCall(trigger.action.outText, string.format("Nuke Agana CAP: exploded %d units", exploded), 10)
  end
end

function WW2PacCVOps:_testNukeGround1()
  local categories = { Group.Category.GROUND }
  local exploded = self:_explodeRedGroupsByPrefix("Ground-1", categories, 250)

  if trigger and trigger.action and trigger.action.outText then
    _safeCall(trigger.action.outText, string.format("Nuke Ground-1: exploded %d units", exploded), 10)
  end
end

function WW2PacCVOps:_testNukeAll()
  local categories = {
    Group.Category.GROUND,
    Group.Category.AIRPLANE,
    Group.Category.HELICOPTER,
    Group.Category.SHIP,
  }

  -- _explodeRedGroupsByPrefix requires a non-empty prefix; for Nuke All, iterate without prefix.
  local exploded = 0
  if coalition and coalition.getGroups then
    for _, category in ipairs(categories) do
      local ok, groups = _safeCall(coalition.getGroups, coalition.side.RED, category)
      if ok and type(groups) == "table" then
        for _, grp in ipairs(groups) do
          exploded = exploded + self:_explodeDCSGroupUnits(grp, 400)
        end
      end
    end
  end

  if trigger and trigger.action and trigger.action.outText then
    _safeCall(trigger.action.outText, string.format("Nuke All: exploded %d red units", exploded), 10)
  end
end

function WW2PacCVOps:_testWinMap()
  local capturedCount = 0
  local total = 0
  local notCaptured = {}

  for _, field in pairs(self.State.Fields or {}) do
    total = total + 1
    -- Clear the airfield area and force it to BLUE with occupation troops.
    self:_applyPersistedBlueField(field)
    if field and field.Captured then
      capturedCount = capturedCount + 1
    else
      table.insert(notCaptured, field and field.Name or "<nil>")
    end
  end

  if trigger and trigger.action and trigger.action.outText then
    local suffix = ""
    if #notCaptured > 0 then
      suffix = " (NOT captured: " .. table.concat(notCaptured, ", ") .. ")"
    end
    _safeCall(trigger.action.outText, string.format("Win map: forced %d/%d airfields to BLUE; checking victory...%s", capturedCount, total, suffix), 12)
  end

  self:_checkVictory()
end

function WW2PacCVOps:_initTestMenu()
  if not (self.Config and self.Config.EnableTestMenu) then
    return
  end
  if not (missionCommands and missionCommands.addSubMenu and missionCommands.addCommand) then
    _warn("missionCommands API not available; test menu not created")
    return
  end

  local root = missionCommands.addSubMenu("WW2PacCVOps TEST")
  missionCommands.addCommand("Nuke Agana", root, function() WW2PacCVOps:_testNukeAgana() end)
  missionCommands.addCommand("Nuke Agana CAP", root, function() WW2PacCVOps:_testNukeAganaCAP() end)
  missionCommands.addCommand("Nuke Ground-1", root, function() WW2PacCVOps:_testNukeGround1() end)
  missionCommands.addCommand("Nuke All", root, function() WW2PacCVOps:_testNukeAll() end)
  missionCommands.addCommand("Win map", root, function() WW2PacCVOps:_testWinMap() end)
end

function WW2PacCVOps:Start()
  if not BASE then
    _err("MOOSE BASE not found. Load Moose_.lua before WW2PacCVOps.lua")
    return
  end

  _log("Starting WW2 Pacific Carrier Ops")

  self.State.StartTime = (timer and timer.getTime) and timer.getTime() or 0
  self.State.CAPEnabled = false

  self:_initFields()
  self:_initConvoys()
  self:_initA2G()
  self:_initLandingHandler()
  self:_initCAPDebugHandlers()
  self:_initTestMenu()

  -- Landing support: artillery + infantry (independent of airfield logic)
  self:_initLandingArtyZones()
  self:_initLandingInfantryRespawns()
  self:_initRandoLandingRespawns()

  -- Apply persisted BLUE airfields after normal init (which defaults all to RED).
  self:_applyPersistenceOnStart()

  -- Draw initial red circles for all Red airfields.
  self:_drawInitialFieldCircles()

  -- Spawn convoys at mission start.
  self:_spawnInitialConvoys()

  -- Spawn A2G bombers at mission start.
  self:_spawnInitialA2G()

  -- Schedule hygiene restart.
  self:_scheduleHygieneRestart()

  -- Delay initial CAP spawn to let late-activated airfield units come alive.
  local function enableAndSpawnCAP()
    self.State.CAPEnabled = true
    _log(string.format("CAP enabled (delay=%ss)", tostring(self.Config.InitialCAPDelaySeconds)))
    self:_spawnInitialCAP()
  end

  -- Periodic ticks.
  if SCHEDULER and SCHEDULER.New then
    self.State.AirfieldScheduler = SCHEDULER:New(nil, function()
      WW2PacCVOps:_tickAirfields()
    end, {}, self.Config.AirfieldTickInitialDelay, self.Config.AirfieldTickSeconds)

    self.State.ConvoyScheduler = SCHEDULER:New(nil, function()
      WW2PacCVOps:_tickConvoys()
      WW2PacCVOps:_tickA2G()
    end, {}, 10, self.Config.RespawnTickSeconds)

    self.State.LandingInfantryScheduler = SCHEDULER:New(nil, function()
      WW2PacCVOps:_tickLandingInfantry()
    end, {}, 5, self.Config.LandingInfantryRespawnTickSeconds)

    self.State.RandoLandingScheduler = SCHEDULER:New(nil, function()
      WW2PacCVOps:_tickRandoLanding()
    end, {}, 5, self.Config.RandoLandingRespawnTickSeconds)
  else
    _warn("MOOSE SCHEDULER not available; falling back to timer.scheduleFunction")

    if timer and timer.scheduleFunction then
      timer.scheduleFunction(function()
        WW2PacCVOps:_tickAirfields()
        return timer.getTime() + WW2PacCVOps.Config.AirfieldTickSeconds
      end, nil, timer.getTime() + self.Config.AirfieldTickInitialDelay)

      timer.scheduleFunction(function()
        WW2PacCVOps:_tickConvoys()
        WW2PacCVOps:_tickA2G()
        return timer.getTime() + WW2PacCVOps.Config.RespawnTickSeconds
      end, nil, timer.getTime() + 10)

      timer.scheduleFunction(function()
        WW2PacCVOps:_tickLandingInfantry()
        return timer.getTime() + (WW2PacCVOps.Config.LandingInfantryRespawnTickSeconds or 5)
      end, nil, timer.getTime() + 5)

      timer.scheduleFunction(function()
        WW2PacCVOps:_tickRandoLanding()
        return timer.getTime() + (WW2PacCVOps.Config.RandoLandingRespawnTickSeconds or 5)
      end, nil, timer.getTime() + 5)
    end
  end

  -- One-shot CAP enable/spawn (use DCS timer even if MOOSE SCHEDULER exists).
  if timer and timer.scheduleFunction and timer.getTime then
    timer.scheduleFunction(function()
      enableAndSpawnCAP()
      return nil
    end, nil, timer.getTime() + (self.Config.InitialCAPDelaySeconds or 60))
  else
    enableAndSpawnCAP()
  end

  _log("WW2PacCVOps initialized")
end

-- Auto-start.
WW2PacCVOps:Start()

return WW2PacCVOps
