// =====================
// NVG COLOR PROFILES
// =====================
// 0 = Green NVG
// 1 = White / Phosphor NVG
// 2 = Blue NVG
// 3 = Red NVG
// =====================


// =====================
// KULLANIM / USAGE
// =====================
//
// 🇹🇷 TÜRKÇE:
// Sadece aşağıdaki satırdaki sayıyı değiştirip dosyayı KAYDEDİNİZ.
//
// #define NVG_COLOR_MODE 0
//
// Renk seçimi:
// 0 → Yeşil NVG
// 1 → Beyaz (Phosphor) NVG
// 2 → Mavi NVG
// 3 → Kırmızı NVG
//
// Dosyayı kaydettikten sonra:
// - OVGME / JSGME modunu Disable → Enable yapınız
// - Ardından DCS World'ü çalıştırınız
//
// --------------------------------------------------
//
// 🇬🇧 ENGLISH:
// Simply change the number below and SAVE the file.
//
// #define NVG_COLOR_MODE 0
//
// Color selection:
// 0 → Green NVG
// 1 → White / Phosphor NVG
// 2 → Blue NVG
// 3 → Red NVG
//
// After saving the file:
// - Disable → Enable the mod in OVGME / JSGME
// - Then launch DCS World
//
// =====================

#define NVG_COLOR_MODE 2   // <<< SADECE BURAYI DEĞİŞTİR / CHANGE ONLY THIS

/* =========================
   ASIL TURQ NVG CORE PRESET
   ========================= */

// 🇹🇷 NVG parlaklık kazancı
// 🇬🇧 NVG brightness gain
float NVG_GAIN = 1.7;     // 1.4 – 2.2

// 🇹🇷 NVG blur miktarı
// 🇬🇧 NVG blur amount
float BLUR_STR = 0.06;    // 0.00 – 0.15

// 🇹🇷 Karıncalanma (film grain)
// 0.00 = Kapalı
// 0.01 = Çok hafif (önerilen)
// 0.02 = Hafif
// 0.03 = Belirgin
//
// 🇬🇧 Film grain / noise
// 0.00 = Disabled
// 0.01 = Very subtle (recommended)
// 0.02 = Light
// 0.03 = Strong
/* ========================= */
float NVG_GRAIN_AMOUNT = 0.02;
/* ========================= */


#include "common/samplers11.hlsl"
#include "common/states11.hlsl"
#include "common/context.hlsl"
#include "deferred/blur.hlsl"
#include "noise/noise3D.hlsl"
#include "NVD_common.hlsl"

Texture2D DiffuseMap;
Texture2D NVDMap;
Texture2D DepthMap;

uint2  dims;
float4 viewport;
float3 color;
float  gain;
float  noiseFactor;

static const float3 LUM = { 1.0, 0.0721f, 0.0721f };

/* =========================
   FULLSCREEN QUAD
   ========================= */

struct VS_OUTPUT {
    float4 pos     : SV_POSITION;
    float4 projPos : TEXCOORD0;
};

static const float2 quad[4] = {
    {-1, -1}, { 1, -1},
    {-1,  1}, { 1,  1}
};

VS_OUTPUT VS(uint vid : SV_VertexID)
{
    VS_OUTPUT o;
    o.projPos = o.pos = float4(quad[vid], 0, 1);
    return o;
}

/* =========================
   COMPOSE PASS (MASK)
   ========================= */

float4 PS(const VS_OUTPUT i, uniform bool useMask, uniform bool useTech2) : SV_TARGET0
{
    float2 uv = float2(
        i.projPos.x * 0.5 + 0.5,
       -i.projPos.y * 0.5 + 0.5
    ) * viewport.zw + viewport.xy;

    float3 nvg = NVDMap.SampleLevel(ClampPointSampler, uv, 0).rgb;

    if (!useMask)
        return float4(nvg, 1);

    float3 base = DiffuseMap.SampleLevel(ClampPointSampler, uv, 0).rgb;

    float m0, m1;

    if (useTech2)
    {
        float2 d = calcMaskCoord2(i.projPos);
        m0 = 1.0 - getMask2(d.x * (1.0 + 0.15 * d.y / d.x), 3.0 * d.x / d.y);
        m1 = getMask2(d.x, 10.0 * d.x / d.y);
    }
    else
    {
        float2 uvm = calcMaskCoord(i.projPos);
        m0 = 1.0 - getMask(uvm * 0.62, 2.4);
        m1 = getMask(uvm, 8.0);
    }

    return float4(m0 * base + m1 * nvg, 1);
}

/* =========================
   BLUR
   ========================= */

#define FOCUS_DISTANCE 10.0

float3 BlurOffs(const VS_OUTPUT i, float2 offs, out float depth)
{
    float2 uv = float2(
        i.projPos.x * 0.5 + 0.5,
       -i.projPos.y * 0.5 + 0.5
    );

    depth = DepthMap.SampleLevel(gBilinearClampSampler, uv, 0).r;
    float4 pos = mul(float4(i.projPos.xy, depth, 1), gProjInv);

    float sigma = 0.8 + 2.0 * saturate((FOCUS_DISTANCE - (pos.z / pos.w)) / FOCUS_DISTANCE);
    return Blur(uv, offs * (0.5 / dims) * BLUR_STR, sigma, NVDMap);
}

float4 PS_BlurX(const VS_OUTPUT i) : SV_TARGET0
{
    float depth;
    return float4(BlurOffs(i, float2(1, 0), depth), 1);
}

/* =========================
   SIMPLE NOISE (FILM GRAIN)
   ========================= */

float hash31(float3 p)
{
    p = frac(p * float3(0.1031, 0.11369, 0.13787));
    p += dot(p, p.yzx + 19.19);
    return frac((p.x + p.y) * p.z);
}

float noise2D(float2 p, float t)
{
    t += sin(p.x * 20) + cos(p.y * 20);
    float i = floor(t);
    float f = frac(t);
    return lerp(hash31(float3(p, i)), hash31(float3(p, i + 1)), f);
}

/* =========================
   FINAL NVG PASS
   ========================= */

float4 PS_BlurY(const VS_OUTPUT i) : SV_TARGET0
{
    float depth;
    float3 base = BlurOffs(i, float2(0, 1), depth);

    // NVG luma (gain tek yerden)
    float nvg_luma = dot(base, LUM) * NVG_GAIN;

    float3 result;

#if NVG_COLOR_MODE == 0
    result = nvg_luma * float3(0.10, 1.00, 0.10);   // Green
#elif NVG_COLOR_MODE == 1
    result = nvg_luma * float3(0.95, 0.95, 0.90);   // White
#elif NVG_COLOR_MODE == 2
    result = nvg_luma * float3(0.35, 0.55, 1.00);   // Blue
#elif NVG_COLOR_MODE == 3
    result = nvg_luma * float3(1.00, 0.20, 0.20);   // Red
#endif

    // Micro-contrast (pseudo sharpen)
    float luma = dot(result, LUM);
    result *= (0.85 + 0.35 * saturate(luma));

    // Hafif karıncalanma (film grain)
    if (NVG_GRAIN_AMOUNT > 0.0)
    {
        float g = noise2D((i.projPos.xy * 0.5 + 0.5) * 120, gModelTime * 1.5);
        g -= 0.5;
        result += result * g * (NVG_GRAIN_AMOUNT * 2.0);
    }

    return float4(result, 1);
}

/* =========================
   TECHNIQUES
   ========================= */

#define COMMON_PART \
    SetVertexShader(CompileShader(vs_5_0, VS())); \
    SetGeometryShader(NULL); \
    SetDepthStencilState(disableDepthBuffer, 0); \
    SetBlendState(disableAlphaBlend, float4(0,0,0,0), 0xFFFFFFFF); \
    SetRasterizerState(cullNone);

technique10 Compose {
    pass P0 { SetPixelShader(CompileShader(ps_5_0, PS(false, false))); COMMON_PART }
    pass P1 { SetPixelShader(CompileShader(ps_5_0, PS(true,  false))); COMMON_PART }
    pass P2 { SetPixelShader(CompileShader(ps_5_0, PS(true,  true ))); COMMON_PART }
}

technique10 BlurX {
    pass P0 { SetPixelShader(CompileShader(ps_5_0, PS_BlurX())); COMMON_PART }
}

technique10 BlurY {
    pass P0 { SetPixelShader(CompileShader(ps_5_0, PS_BlurY())); COMMON_PART }
}
