'use strict'

/*
 *  Constants
 */
// the default width of a line
const DEF_STROKE_WIDTH    = 0.7;

// the default radius of a circle around the points
const DEF_POINT_RADIUS		= 6;

// whether the calipers start with range display on or off
const DEF_CALIPER_CHEAT_MODE	= true;

// number of layers
const LAYER_COUNT					= 4;

// default name of the data file when saved
const DEF_FILE_NAME				= "plot.text";

// possible states of the program
const DEFAULT_STATE				= 0;
const ROTATING_STATE			= 1;
const POINTING_STATE			= 2;
const LINING_STATE        = 3;
const TEXTING_STATE       = 4;
const SELECTING_STATE     = 5;
const COMPUTING_STATE     = 6;
const COMPUTER_ROT_STATE  = 7;
const CAL_MOVE_STATE		  = 8;
const CAL_SPREAD_STATE	  = 9;
const HELPING_STATE			  = 10;

// possibly useful colors, may or may not be used
const BLACK_COLOR					= "rgb(0 0 0)";
const RED_COLOR						= "rgb(255 0 0)";
const GREEN_COLOR					= "rgb(0 255 0)";
const BLUE_COLOR					= "rgb(0 0 255)";
const PENCIL_COLOR        = "rgb(100, 100, 100)";
const SEL_TEXT_COLOR      = "rgb(200, 0, 0)";

// attributes of the buttons used to contol the active layer, sizes are relative to screen size
const BUTTONS_TOP         = 1.2; // top and left of the first button
const BUTTONS_LEFT        = 1.2;
const BUTTONS_WIDTH       = 2.3; // width of the button
const BUTTONS_HEIGHT      = BUTTONS_WIDTH * 1.3;
const BUTTONS_SPACING     = 0.6;

// characters which can be typed in a text object
const ALLOWED_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890 -/\"'#&(),.:;?!@$%^*<>_°{}[]+=\\|\n";

// used to recognize a valid file when loading a plot
const PROGRAM_NAME        = "Plotter.js"

/*
 *  Classes
 */
class Point {
	constructor(x, y) {
		this.type			= "point";
		this.rel_x		= toRelPos(x);
		this.rel_y		= toRelPos(y);
		this.layer		= -1;
		this.visible	= true;
	}
}

class Line {
	constructor(start_x, start_y, end_x, end_y) {
		this.type					= "line";
		this.rel_start_x	= toRelPos(start_x);
		this.rel_start_y	= toRelPos(start_y);
		this.rel_end_x		= toRelPos(end_x);
		this.rel_end_y		= toRelPos(end_y);
		this.start_set		= false;
		this.layer				= -1;
		this.visible			= true;
		this.tick_dist		= 0; // the distance in miles between ticks on a line
	}
}

class Text {
	constructor(text, x, y) {
		this.type			= "text";
		this.text			= text;
		this.rel_x		= toRelPos(x);
		this.rel_y		= toRelPos(y) - 0.01; // offset so the text isn't under the mouse cursor
		this.clean		= true;
		this.layer		= -1;
		this.visible	= true;
	}
}

class Calipers {
	constructor() {
		this.show						= false;
		this.showPosition		= false;
		this.x							= 0.45;
		this.y							= 0.2;
		this.offset					= 0.1;
		this.len						= 0.2;
		this.center_lat			= null;
		this.center_long		= null;
		this.m_down_x				= 0;
		this.m_down_y				= 0;
		this.cheat_mode			= DEF_CALIPER_CHEAT_MODE;
		this.lat_disp				= document.getElementById("lat-disp");
		this.long_disp			= document.getElementById("long-disp");
	}

	display() {
		if (this.show) {
			let a = graph_angle + angle_delta;
			let [s_x, s_y] = rotatePoint(this.x, this.y, a);
			let [e_x, e_y] = rotatePoint(this.x, this.y + this.len, a);
			let [t1_x, t1_y] = rotatePoint(this.x-0.005, this.y-0.005, a);
			let [t2_x, t2_y] = rotatePoint(this.x+0.005, this.y-0.005, a);
			let [t3_x, t3_y] = rotatePoint(this.x-0.005, this.y+this.len+0.005, a);
			let [t4_x, t4_y] = rotatePoint(this.x+0.005, this.y+this.len+0.005, a);

			// draw the base line
			drawLine(null, {
				start_x:	toAbsPos(s_x),
				start_y:	toAbsPos(s_y),
				end_x:		toAbsPos(e_x),
				end_y:		toAbsPos(e_y),
				color:		BLUE_COLOR,
			});

			// draw the top triangle
			drawLine(null, {
				start_x:	toAbsPos(s_x),
				start_y:	toAbsPos(s_y),
				end_x:		toAbsPos(t1_x),
				end_y:		toAbsPos(t1_y),
				color:		BLUE_COLOR,
			});
			drawLine(null, {
				start_x:	toAbsPos(t1_x),
				start_y:	toAbsPos(t1_y),
				end_x:		toAbsPos(t2_x),
				end_y:		toAbsPos(t2_y),
				color:		BLUE_COLOR,
			});
			drawLine(null, {
				start_x:	toAbsPos(t2_x),
				start_y:	toAbsPos(t2_y),
				end_x:		toAbsPos(s_x),
				end_y:		toAbsPos(s_y),
				color:		BLUE_COLOR,
			});
			
			// draw the bottom triangle
			drawLine(null, {
				start_x:	toAbsPos(e_x),
				start_y:	toAbsPos(e_y),
				end_x:		toAbsPos(t3_x),
				end_y:		toAbsPos(t3_y),
				color:		BLUE_COLOR,
			});
			drawLine(null, {
				start_x:	toAbsPos(t3_x),
				start_y:	toAbsPos(t3_y),
				end_x:		toAbsPos(t4_x),
				end_y:		toAbsPos(t4_y),
				color:		BLUE_COLOR,
			});
			drawLine(null, {
				start_x:	toAbsPos(t4_x),
				start_y:	toAbsPos(t4_y),
				end_x:		toAbsPos(e_x),
				end_y:		toAbsPos(e_y),
				color:		BLUE_COLOR,
			});

			// draw the end line
			[s_x, s_y] = rotatePoint(this.x+this.offset, this.y, a);
			[e_x, e_y] = rotatePoint(this.x+this.offset, this.y+this.len, a);
			drawLine(null, {
				start_x:	toAbsPos(s_x),
				start_y:	toAbsPos(s_y),
				end_x:		toAbsPos(e_x),
				end_y:		toAbsPos(e_y),
				color:		BLUE_COLOR,
			});

			if (this.cheat_mode) {
				let s = scale_factors[data.scales[data.active_layer]]
				// 332.19 is the scale factor between the relative size of the window and miles on the 150 NM graph
				range_disp.value = Math.abs(Math.round(" " + this.offset * 332.19 * s + " "));
				range_disp.style.display = "block";
			} else {
				range_disp.style.display = "none";
			}
		} else {
			range_disp.style.display = "none";
		}
	}

	distanceToBase(x, y) {
		let [r_x, r_y] = rotatePoint(toRelPos(x), toRelPos(y), -(graph_angle + angle_delta));
		return distanceToLine(null, r_x, r_y, this.x, this.y, this.x, this.y + this.len);
	}

	distanceToEnd(x, y) {
		let [r_x, r_y] = rotatePoint(toRelPos(x), toRelPos(y), -(graph_angle + angle_delta));
		return distanceToLine(null, r_x, r_y, this.x+this.offset, this.y, this.x+this.offset, this.y + this.len);
	}

	setMouseDown(x, y) {
		let [r_x, r_y] = rotatePoint(toRelPos(x), toRelPos(y), -(graph_angle + angle_delta));
		this.m_down_x = this.x - r_x;
		this.m_down_y = this.y - r_y;
	}

	setPosition(x, y) {
		let [r_x, r_y] = rotatePoint(toRelPos(x), toRelPos(y), -(graph_angle + angle_delta));
		this.x = r_x + this.m_down_x;
		this.y = r_y + this.m_down_y;
	}

	setOffset(x, y) {
		let [r_x, r_y] = rotatePoint(toRelPos(x), toRelPos(y), -(graph_angle + angle_delta));
		this.offset = r_x - this.x;
	}

	setRange(r) {
		// 332.19 is the scale factor between the relative size of the window and miles on the 150 NM graph
		let s = scale_factors[data.scales[data.active_layer]]
		if (this.offset >= 0) {
			this.offset = r / 332.19 / s;
		} else {
			this.offset = -r / 332.19 / s;
		}
		drawObjects();
	}

	setLatitude() {
		// set the width of the calipers to the size of a degree of longitude at the latitude the graph is turned to
		// 332.19 is the scale factor between the relative size of the window and miles on the 150 NM graph
		let a = (graph_angle + angle_delta);
		let s = scale_factors[data.scales[data.active_layer]]
		if (this.offset >= 0) {
			this.offset = (Math.cos(a) * 60 / 332.19) / s;
		} else {
			this.offset = -(Math.cos(a) * 60 / 332.19) / s;
		}
		drawObjects();
	}
	
	setCenter() {
		if (this.cheat_mode === true) {
			this.center_lat = Math.round(graph_angle * 180.0 / Math.PI);
			this.center_long = Math.round(range_disp.value);
			if (this.offset < 0) {
				this.center_long = -this.center_long;
			}
			this.lat_disp.textContent = "";
			this.lat_disp.style.display = "block";
			this.long_disp.textContent = "";
			this.long_disp.style.display = "block";
			this.showPosition = true;
		}
	}

	getCurrentPos() {
		if (distanceToPoint(null, toRelPos(global_mouse.x), toRelPos(global_mouse.y), 0.5, 0.5) > 0.51) {
			// return with no display if the cursor is outside the graph
			this.lat_disp.textContent = "";
			this.long_disp.textContent = "";
			return;
		}

		if ((this.center_lat !== null) && (this.showPosition === true)) {
			let nm_offset;
			let degrees;
			let minutes;
			let hemi;

			// the scale of the current graph
			const s = scale_factors[data.scales[data.active_layer]]

			// scale factor for the latitude
			const l_s = Math.cos(this.center_lat * Math.PI / 180.0);

			// get the latitude
			nm_offset = (this.center_lat * 60.0) - ((toRelPos(global_mouse.y) - 0.5) * 332.19 * s);
			degrees = Math.floor(Math.abs(nm_offset / 60.0));
			minutes = Math.abs(Math.round(nm_offset % 60));
			if (minutes === 60) {
				minutes = 0;
				degrees += 1;
			}
			hemi = (nm_offset >= 0) ? 'N' : 'S';
			this.lat_disp.textContent = `${String(degrees).padStart(2, '0')}° ${String(minutes).padStart(2, '0')}' ${hemi}`;

			// get the longitude
			nm_offset = (this.center_long * 60) + (((toRelPos(global_mouse.x) - 0.5) * 332.19 * s) / l_s);
			degrees = Math.floor(Math.abs(nm_offset / 60.0));
			minutes = Math.abs(Math.round(nm_offset % 60));
			if (degrees > 179) {
				degrees = 359 - degrees;
				minutes = 60 - minutes;
				nm_offset = -nm_offset;
			}
			if (minutes === 60) {
				minutes = 0;
				degrees += 1;
			}
			hemi = (nm_offset >= 0) ? 'E' : 'W';
			this.long_disp.textContent = `${String(degrees).padStart(3, '0')}° ${String(minutes).padStart(2, '0')}' ${hemi}`; 
		}
	}
}

/*
 *  Global variables
 */
let global_state = [DEFAULT_STATE];	// current state of the program
let graph_angle = 0;								// angle the graph is rotated to
let computer_angle = 0;							// angle the inner computer ring is rotated to
let angle_delta = 0;								// amount the graph or computer has been rotated during active rotation
let mouse_down_angle = 0;						// angle from the center when we start rotating
let filename = DEF_FILE_NAME;				// the filename to use when saving

// the canvas element used to draw the point, line, and text objects
const canvas = document.getElementById("canvas");

// context for the canvas
const ctx = canvas.getContext("2d");

// graphs for three different scales
const graph_elements = [
	document.getElementById("graph150"),
	document.getElementById("graph200"),
	document.getElementById("graph300"),
];

// maps for three different scales
const map_elements = [
	document.getElementById("map150"),
	document.getElementById("map200"),
	document.getElementById("map300"),
];

// displays the distance between the caliper lines, used for measuring and entering data
const range_disp = document.getElementById("caliperRange");

// deleted elements stored for undo
const deleted_objects = [];

// object to control calipers
const calipers = new Calipers();

// the current mouse position, set from the handleMouseMove function
const global_mouse = {
  x: 0,
  y: 0,
}

// whether to display the map on each layer or not
displayActiveScale.display_map = [];
for (let i=0; i<LAYER_COUNT; i++) {
	displayActiveScale.display_map.push(false);
}

// ratio between the different scales, in order they are 150NM, 200NM, and 300NM
const scale_factors = [1, 4/3, 2];

// display crosshairs to aid in positioning
displayCrosshairs.show = false;

// find the center of the graph, used as the rotation center
// assumes the graph is square, the height of the document, and located at (0, 0)
const graph_center = {
	x: document.documentElement.clientHeight / 2,
	y: document.documentElement.clientHeight / 2,
};

// stroke width of circles and lines
let stroke_width = DEF_STROKE_WIDTH;

// radius of the circle surrounding a point, in pixels
let point_radius = DEF_POINT_RADIUS;

// used in selecting items
// saves the two objects closest to the cursor
const selected = {
	_index:		0,
	object_index:	[null, null],

	get index() {
		return this.object_index[this._index];
	},

	swap() {
		if (this._index === 0) {
			if (this.object_index[1] !== null) {
				this._index = 1;
			}
		} else {
			this._index = 0;
		}
	},

	reset() {
		this._index = 0;
		this.object_index[0] = null;
		this.object_index[1] = null;
	}
};

// keep track of whether the plot has been changed
const file = {
	_dirty:	false,

	get dirty() {
		return this._dirty;
	},

	set dirty(d) {
		this._dirty = d;
		if (this._dirty) {
			document.title = `Mk. 3 Plotter (${filename}*)`;
			addEventListener("beforeunload", handleUnload);
		} else {
			document.title = `Mk. 3 Plotter (${filename})`;
			removeEventListener("beforeunload", handleUnload);
		}
	},
};

let read_data = null; // used to read data from a file

// the main store of information for the plot, this is what's saved to disk
let data = {
  program_name:	PROGRAM_NAME,
	active_layer:	0,
	scales:				[],
	objects:			[],
};

// initialize visibility for each layer, and the buttons to contol it
for (let i=0; i<LAYER_COUNT; i++) {
	data.scales.push(0); // start with every layer set to the 150 NM scale

  // make the buttons to control active layer
	let button = document.createElement('button');
  button.className = "layer-buttons";
  button.style.top = (BUTTONS_TOP + ((BUTTONS_HEIGHT + BUTTONS_SPACING) * i)) + "vh";
  button.style.left = BUTTONS_LEFT + "vh";
  button.style.width = BUTTONS_WIDTH + "vh";
  button.id = i;
  button.textContent = (i === data.active_layer) ? 'X' : '';
  button.addEventListener("click", handleButtonClick);
  document.getElementById("buttons").appendChild(button);
}

/*
 *  Initialize things
 */
// don't allow scrolling, it messes things up
document.body.style.overflow = "hidden";

// set the canvas to the viewport height and width
// for some reason the canvas needs to be resized whenever the window changes size
canvas.width = document.documentElement.clientWidth;
canvas.height = document.documentElement.clientHeight;

// set the document title
document.title = `Mk. 3 Plotter (${filename})`;

// show the correct scale for the active layer
displayActiveScale();

// set the text color
let CSS_vars = document.querySelector(":root");
CSS_vars.style.setProperty("--text-color", PENCIL_COLOR);

/*
 *  Add event handlers
 */
addEventListener("mousedown", handleMouseDown);
addEventListener("mouseup", handleMouseUp);
addEventListener("mousemove", handleMouseMove);
addEventListener("contextmenu", handleRightClick);
addEventListener("keydown", handleKeyDown);
addEventListener("resize", handleResize);
document.getElementById("loadButton").addEventListener("change", handleFileChange);
document.getElementById("saveButton").addEventListener("click", handleSaveButton);
document.getElementById("caliperRange").addEventListener("input", handleRangeInput);

/*
 *  Callback functions
 */
function handleMouseDown(event) {
  let mouse_down_x = 0; // these are used for calculations below
  let mouse_down_y = 0;

	switch (global_state.at(-1)) {
		case DEFAULT_STATE:
			if (event.button === 0) {
				mouse_down_x = global_mouse.x - graph_center.x;
				mouse_down_y = global_mouse.y - graph_center.y;
				mouse_down_angle = Math.atan2(mouse_down_y, mouse_down_x);
				if (calipers.show)
					if (calipers.distanceToEnd(global_mouse.x, global_mouse.y) < 0.005) {
						calipers.setMouseDown(global_mouse.x, global_mouse.y);
						global_state.push(CAL_SPREAD_STATE);
					} else if (calipers.distanceToBase(global_mouse.x, global_mouse.y) < 0.005) {
						calipers.setMouseDown(global_mouse.x, global_mouse.y);
						global_state.push(CAL_MOVE_STATE);
					} else {
						global_state.push(ROTATING_STATE);
					}
				else {
					global_state.push(ROTATING_STATE);
				}
      }
			break;

		case POINTING_STATE:
      if (distanceToPoint(data.objects.at(-1)) < 0.51) {
        data.objects.at(-1).layer = data.active_layer;
      }
			document.body.style.cursor = "auto";
			file.dirty = true;
			global_state.pop();
      if (global_state.at(-1) === SELECTING_STATE) {
        data.objects.splice(selected.index, 1);
				selectClosest();
      }
			break;

		case LINING_STATE:
			if (event.button === 0) {
        if (data.objects.at(-1).start_set) {
          // set_start is true, set the end position
          data.objects.at(-1).rel_end_x = toRelPos(event.x);
          data.objects.at(-1).rel_end_y = toRelPos(event.y);
          if (distanceToLine(data.objects.at(-1)) < 0.51) {
            data.objects.at(-1).layer = data.active_layer;
					}
          document.body.style.cursor = "auto";
					file.dirty = true;
          global_state.pop();
          if (global_state.at(-1) === SELECTING_STATE) {
            data.objects.splice(selected.index, 1);
						selectClosest();
          } else {
						drawObjects();
					}
        } else {
          // set_start is false, set the start position
          data.objects.at(-1).rel_start_x = toRelPos(event.x);
          data.objects.at(-1).rel_start_y = toRelPos(event.y);
          data.objects.at(-1).start_set = true;
          data.objects.at(-1).visible = true;
          document.body.style.cursor = "none";
          if (global_state.at(-2) === SELECTING_STATE) {
						if (distanceToLine(data.objects.at(-1)) < 0.51) {
							data.objects.at(-1).layer = data.active_layer;
						}
            data.objects.splice(selected.index, 1);
            document.body.style.cursor = "auto";
						file.dirty = true;
            global_state.pop();
						selectClosest();
					}
        }
      }
			break;

		case TEXTING_STATE:
			if (event.button === 0) {
        // mouse down texting
        if (data.objects.at(-1).clean === true) {
          data.objects.pop();
          if (global_state.at(-2) === SELECTING_STATE) {
            data.objects[selected.index].visible = true;
						selectClosest();
          } else {
						drawObjects();
					}
        } else {
          if (distanceToText(data.objects.at(-1)) < 0.51) {
            data.objects.at(-1).layer = data.active_layer;
          }
					file.dirty = true;
          if (global_state.at(-2) === SELECTING_STATE) {
            data.objects.splice(selected.index, 1);
						selectClosest();
          } else {
						drawObjects();
					}
        }
        global_state.pop();
      }
			break;

		case SELECTING_STATE:
			if (event.button === 0) {
				selected.swap();
				drawObjects();
				drawObject(data.objects[selected.index], { color:RED_COLOR, line_width:stroke_width + 0.4, });
			}
			break;

		case COMPUTING_STATE:
			if (event.button === 0) {
				mouse_down_x = global_mouse.x - graph_center.x;
				mouse_down_y = global_mouse.y - graph_center.y;
				mouse_down_angle = Math.atan2(mouse_down_y, mouse_down_x);
				global_state.push(COMPUTER_ROT_STATE);
      }
			break;
	}
}

function handleRightClick(event) {
  event.preventDefault();

	switch (global_state.at(-1)) {
		case DEFAULT_STATE:
			if(selectClosest()) {
				// found something so go to selecting state
				global_state.push(SELECTING_STATE);
			}
			break;

		case SELECTING_STATE:
			selected.reset();
			drawObjects();
			global_state.pop();
			break;
	}
}

function handleMouseUp(event) {
	switch (global_state.at(-1)) {
		case ROTATING_STATE:
			graph_angle = graph_angle + angle_delta;
			angle_delta = 0;
			global_state.pop();
			break;

		case CAL_MOVE_STATE:
			global_state.pop();
			break;

		case CAL_SPREAD_STATE:
			global_state.pop();
			break;

		case COMPUTER_ROT_STATE:
			computer_angle = computer_angle + angle_delta;
			angle_delta = 0;
      global_state.pop();
			break;
	}
}

function handleMouseMove(event) {
  let mouse_x = 0;
  let mouse_y = 0;

  global_mouse.x = event.clientX;
  global_mouse.y = event.clientY;

	if ((displayCrosshairs.show) || (calipers.show)) {
		drawObjects();
	}

	calipers.getCurrentPos();

	switch (global_state.at(-1)) {
		case ROTATING_STATE:
			mouse_x = global_mouse.x - graph_center.x;
			mouse_y = global_mouse.y - graph_center.y;
			angle_delta = Math.atan2(mouse_y, mouse_x) - mouse_down_angle;
			for (let i=0; i<3; i++) {
				graph_elements[i].style.transform = 'rotate(' + (graph_angle + angle_delta) + 'rad)';
			}
			drawObjects();
			break;

		case CAL_MOVE_STATE:
			calipers.setPosition(global_mouse.x, global_mouse.y);
			drawObjects();
			break;

		case CAL_SPREAD_STATE:
			calipers.setOffset(global_mouse.x, global_mouse.y);
			drawObjects();
			break;

		case POINTING_STATE:
			data.objects.at(-1).rel_x = toRelPos(global_mouse.x);
			data.objects.at(-1).rel_y = toRelPos(global_mouse.y);
			drawObjects();
			break;

		case LINING_STATE:
			if (data.objects.at(-1).start_set) {
        data.objects.at(-1).rel_end_x = toRelPos(global_mouse.x);
        data.objects.at(-1).rel_end_y = toRelPos(global_mouse.y);
      } else {
        data.objects.at(-1).rel_start_x = toRelPos(global_mouse.x);
        data.objects.at(-1).rel_start_y = toRelPos(global_mouse.y);
      }
			drawObjects();
			break;

		case TEXTING_STATE:
			data.objects.at(-1).rel_x = toRelPos(global_mouse.x);
			data.objects.at(-1).rel_y = toRelPos(global_mouse.y) - 0.01;
			drawObjects();
			break;

		case SELECTING_STATE:
      if(!selectClosest()) {
        // nothing was found
        global_state.pop();
      }
			break;

		case COMPUTER_ROT_STATE:
			mouse_x = global_mouse.x - graph_center.x;
			mouse_y = global_mouse.y - graph_center.y;
			angle_delta = Math.atan2(mouse_y, mouse_x) - mouse_down_angle;
			mk8_inner.style.transform = 'rotate(' + (computer_angle + angle_delta) + 'rad)';
			break;
	}
}

function handleKeyDown(event) {
	switch (global_state.at(-1)) {
		case DEFAULT_STATE:
			switch (event.key) {
        case 'ArrowUp':
          stroke_width += 0.1;
          drawObjects();
          break;

        case 'ArrowDown':
          stroke_width -= 0.1;
          if (stroke_width < 0.1) { stroke_width = 0.1 };
          drawObjects();
          break;

				case 'ArrowRight':
					point_radius += 1;
					drawObjects();
					break;

				case 'ArrowLeft':
					point_radius -= 1;
					if (point_radius < 0) { point_radius = 0; }
					drawObjects();
					break;

				case 'h':
				case 'H':
				case '?':
					document.getElementById("help-div").style.display = "block";
					global_state.push(HELPING_STATE);
					break;

				case 'x':
				case 'X':
          // display the crosshairs
					if (!event.ctrlKey) {
						displayCrosshairs.show = !displayCrosshairs.show;
						drawObjects();
					}
					break;

				case 's':
				case 'S':
					if (event.ctrlKey) {
						event.preventDefault();
						saveButton.click();
					} else {
						// change the scale
						changeActiveScale();
						displayActiveScale();
						drawObjects();
					}
					break;

				case 'm':
				case 'M':
					displayActiveScale.display_map[data.active_layer] = !displayActiveScale.display_map[data.active_layer];
					displayActiveScale();
					break;

				case 'p':
				case 'P':
          // start a point
					if (!event.ctrlKey) {
						document.body.style.cursor = "none";
						data.objects.push(new Point(global_mouse.x, global_mouse.y));
						drawObjects();
						global_state.push(POINTING_STATE);
					}
					break;

				case 'l':
				case 'L':
					if (event.ctrlKey) {
						event.preventDefault();
						loadButton.click();
					} else {
						// start a line
						document.body.style.cursor = "crosshair";
						data.objects.push(new Line(0, 0, 0, 0));
						data.objects.at(-1).visible = false;
						global_state.push(LINING_STATE);
					}
					break;

				case 't':
				case 'T':
          // start a text
					if (!event.ctrlKey) {
						data.objects.push(new Text("Enter text", global_mouse.x, global_mouse.y));
						drawObjects();
						global_state.push(TEXTING_STATE);
					}
					break;

				case 'c':
				case 'C':
          // display the computer
					if (event.ctrlKey) {
						event.preventDefault();
						calipers.setCenter();
					} else {
						mk8_outer.style.display = "block";
						mk8_inner.style.display = "block";
						global_state.push(COMPUTING_STATE);
					}
					break;

				case 'k':
				case 'K':
					if (event.ctrlKey) {
						event.preventDefault();
						calipers.cheat_mode = !calipers.cheat_mode;
					} else {
						calipers.show = !calipers.show;
					}
					drawObjects();
					break;

				case 'w':
				case 'W':
					if (event.ctrlKey) {
						event.preventDefault();
						event.stopPropagation();
						calipers.setLatitude();
					}
					break;

				case 'u':
				case 'U':
					// undelete most recent deletion
					if (deleted_objects.length > 0){
						data.objects.push(deleted_objects.at(-1));
						deleted_objects.pop();
						drawObjects();
					}
					break;

				case 'z':
				case 'Z':
					// undelete most recent deletion
					if (event.ctrlKey && (deleted_objects.length > 0)) {
						event.preventDefault();
						data.objects.push(deleted_objects.at(-1));
						deleted_objects.pop();
						drawObjects();
					}
					break;
			}
			break; // case DEFAULT_STATE
		
		case HELPING_STATE:
			switch(event.key) {
				case 'h':
				case 'H':
				case '?':
				case "Escape":
					document.getElementById("help-div").style.display = "none";
					global_state.pop();
					break;
			}
			break;

		case POINTING_STATE:
			switch (event.key) {
				case "Escape":
					data.objects.pop();
					document.body.style.cursor = "auto";
					drawObjects();
					global_state.pop();
          if (global_state.at(-1) === SELECTING_STATE) {
            data.objects[selected.index].visible = true;
            selectClosest();
          }
					break;

        case 'ArrowUp':
          stroke_width += 0.1;
          drawObjects();
          break;

        case 'ArrowDown':
          stroke_width -= 0.1;
          if (stroke_width < 0.1) { stroke_width = 0.1 };
          drawObjects();
          break;

				case 'ArrowRight':
					point_radius += 1;
					drawObjects();
					break;

				case 'ArrowLeft':
					point_radius -= 1;
					if (point_radius < 0) { point_radius = 0; }
					drawObjects();
					break;

				case 'x':
				case 'X':
          // display the crosshairs
					if (!event.ctrlKey) {
						displayCrosshairs.show = !displayCrosshairs.show;
						drawObjects();
					}
					break;
			}
			break; // case POINTING_STATE
		
		case LINING_STATE:
			switch (event.key) {
				case "Escape":
          data.objects.pop();
					document.body.style.cursor = "auto";
					drawObjects();
					global_state.pop();
          if (global_state.at(-1) === SELECTING_STATE) {
            data.objects[selected.index].visible = true;
						selectClosest();
          }
					break;

        case 'ArrowUp':
          stroke_width += 0.1;
          drawObjects();
          break;

        case 'ArrowDown':
          stroke_width -= 0.1;
          if (stroke_width < 0.1) { stroke_width = 0.1 };
          drawObjects();
          break;

				case 'ArrowRight':
					point_radius += 1;
					drawObjects();
					break;

				case 'ArrowLeft':
					point_radius -= 1;
					if (point_radius < 0) { point_radius = 0; }
					drawObjects();
					break;

				case 'x':
				case 'X':
          // display the crosshairs
					if (!event.ctrlKey) {
						displayCrosshairs.show = !displayCrosshairs.show;
						drawObjects();
					}
					break;
			}
			break; // case LINING_STATE
		
		case TEXTING_STATE:
			switch (event.key) {
				case "Escape":
          data.objects.pop();
					global_state.pop();
          if (global_state.at(-1) === SELECTING_STATE) {
            data.objects[selected.index].visible = true;
            selectClosest();
          } else {
						drawObjects();
					}
					break;

        case 'ArrowUp':
          stroke_width += 0.1;
          drawObjects();
          break;

        case 'ArrowDown':
          stroke_width -= 0.1;
          if (stroke_width < 0.1) { stroke_width = 0.1 };
          drawObjects();
          break;

				case 'ArrowRight':
					point_radius += 1;
					drawObjects();
					break;

				case 'ArrowLeft':
					point_radius -= 1;
					if (point_radius < 0) { point_radius = 0; }
					drawObjects();
					break;

				case "Backspace":
				case "Delete":
          if (data.objects.at(-1).clean !== true) {
            data.objects.at(-1).text = data.objects.at(-1).text.slice(0, -1);
            if (data.objects.at(-1).text.length === 0) {
              data.objects.at(-1).clean = true;
              data.objects.at(-1).text = "Enter text";
            }
          }
					drawObjects();
          break;

				default:
          let c = event.key;
          if ((c === '0') && (event.ctrlKey)) {
            c = '°';
          }
          if (ALLOWED_CHARS.includes(event.key)) {
						event.preventDefault();
            if (data.objects.at(-1).clean === true) {
              data.objects.at(-1).clean = false;
              data.objects.at(-1).text = c;
            } else {
              data.objects.at(-1).text += c;
            }
          }
					drawObjects();
					break;
			}
			break; // case TEXTING_STATE
		
		case SELECTING_STATE:
			switch (event.key) {
				case "Escape":
          selected.reset();
					drawObjects();
					global_state.pop();
					break;

        case 'ArrowUp':
          stroke_width += 0.1;
          selectClosest();
          break;

        case 'ArrowDown':
          stroke_width -= 0.1;
          if (stroke_width < 0.1) { stroke_width = 0.1 };
          selectClosest();
          break;

				case 'ArrowRight':
					point_radius += 1;
					selectClosest();
					break;

				case 'ArrowLeft':
					point_radius -= 1;
					if (point_radius < 0) { point_radius = 0; }
					selectClosest();
					break;

				case 'k':
				case 'K':
					if (event.ctrlKey) {
						event.preventDefault();
						calipers.cheat_mode = !calipers.cheat_mode;
					} else {
						calipers.show = !calipers.show;
					}
					selectClosest();
					break;

				case 'x':
				case 'X':
          // display the crosshairs
					if (!event.ctrlKey) {
						displayCrosshairs.show = !displayCrosshairs.show;
						selectClosest();
					}
					break;

        case 'e':
        case 'E':
					if (!event.ctrlKey) {
						switch (data.objects[selected.index].type) {
							case "point":
								data.objects.push(new Point(global_mouse.x, global_mouse.y));
								data.objects[selected.index].visible = false;
								document.body.style.cursor = "none";
								global_state.push(POINTING_STATE);
								break;

							case "line":
								let m_x = toRelPos(global_mouse.x);
								let m_y = toRelPos(global_mouse.y);
								let sl_x = data.objects[selected.index].rel_start_x;
								let sl_y = data.objects[selected.index].rel_start_y;
								let el_x = data.objects[selected.index].rel_end_x;
								let el_y = data.objects[selected.index].rel_end_y;
								data.objects.push(new Line(toAbsPos(sl_x), toAbsPos(sl_y), toAbsPos(el_x), toAbsPos(el_y)));
								data.objects.at(-1).tick_dist = data.objects[selected.index].tick_dist;
								if (distanceToPoint(null, m_x, m_y, sl_x, sl_y) < distanceToPoint(null, m_x, m_y, el_x, el_y)) {
									// pointer closer to start
									data.objects.at(-1).start_set = false;
									data.objects.at(-1).rel_start_x = m_x;
									data.objects.at(-1).rel_start_y = m_y;
								} else {
									// pointer closer to end
									data.objects.at(-1).start_set = true;
									data.objects.at(-1).rel_end_x = m_x;
									data.objects.at(-1).rel_end_y = m_y;
								}
								data.objects[selected.index].visible = false;
								document.body.style.cursor = "none";
								global_state.push(LINING_STATE);
								break;

							case "text":
								data.objects.push(new Text(data.objects[selected.index].text, global_mouse.x, global_mouse.y));
								data.objects.at(-1).clean = data.objects[selected.index].clean;
								data.objects[selected.index].visible = false;
								global_state.push(TEXTING_STATE);
								break;
						}
						drawObjects();
					}
          break;

        case 'h':
        case 'H':
					if ((!event.ctrlKey) && (data.objects[selected.index].type === "line")) {
						data.objects[selected.index].tick_dist = Number(range_disp.value) / 6.0;
						selectClosest();
					}
					break;

        case "Backspace":
        case "Delete":
        case 'd':
          if (selected.index !== null) {
						deleted_objects.push(data.objects[selected.index]);
            data.objects.splice(selected.index, 1);
						file.dirty = true;
          }
          drawObjects();
          if (!selectClosest()) {
            // nothing was found, exit selecting state
            global_state.pop();
          }
          break;

				case 'u':
				case 'U':
					// undelete most recent deletion
					if (deleted_objects.length > 0){
						data.objects.push(deleted_objects.at(-1));
						deleted_objects.pop();
						selectClosest();
					}
					break;

				case 'z':
				case 'Z':
					// undelete most recent deletion
					if (event.ctrlKey && (deleted_objects.length > 0)) {
						data.objects.push(deleted_objects.at(-1));
						deleted_objects.pop();
						selectClosest();
					}
					break;
			}
			break; // case SELECTING_STATE
		
		case COMPUTING_STATE:
			switch (event.key) {
        case 'ArrowUp':
          stroke_width += 0.1;
          drawObjects();
          break;

        case 'ArrowDown':
          stroke_width -= 0.1;
          if (stroke_width < 0.1) { stroke_width = 0.1 };
          drawObjects();
          break;

				case 'ArrowRight':
					point_radius += 1;
					drawObjects();
					break;

				case 'ArrowLeft':
					point_radius -= 1;
					if (point_radius < 0) { point_radius = 0; }
					drawObjects();
					break;

				case 'x':
				case 'X':
          // display the crosshairs
					if (!event.ctrlKey) {
						displayCrosshairs.show = !displayCrosshairs.show;
						drawObjects();
					}
					break;

				case "Escape":
        case 'c':
					if (!event.ctrlKey) {
						mk8_outer.style.display = "none";
						mk8_inner.style.display = "none";
						global_state.pop();
					}
          break;

        case 't':
        case 'T':
          // start a text
					if (!event.ctrlKey) {
						data.objects.push(new Text("Enter text", global_mouse.x, global_mouse.y));
						drawObjects();
						global_state.push(TEXTING_STATE);
					}
          break;
			}
			break; // case COMPUTING_STATE
	}
}

function handleButtonClick(event) {
  if (global_state.at(-1) === DEFAULT_STATE) {
    data.active_layer = Number(this.id);
    for (let i=0; i<LAYER_COUNT; i++) {
      document.getElementById(i).textContent = (i === data.active_layer) ? 'X' : '';
    }
  displayActiveScale();
  drawObjects();
  }
}

function handleResize(event) {
	canvas.width = document.documentElement.clientWidth;
	canvas.height = document.documentElement.clientHeight;

	graph_center.x = document.documentElement.clientHeight / 2;
	graph_center.y = document.documentElement.clientHeight / 2;

	drawObjects();
}

function handleFileChange(event) {
	if (file.dirty) {
		if (!confirm("Current file has been modified.\n\nLoad the new file anyway?")) { return; }
	}
  let reader = new FileReader();
  reader.onload = (event) => {
    read_data = JSON.parse(event.target.result);
    if ((read_data.program_name) && (read_data.program_name === PROGRAM_NAME)) {
			deleted_objects.splice(0, deleted_objects.length);
			data = read_data;
			for (let i=0; i<LAYER_COUNT; i++) {
				document.getElementById(i).textContent = (i === data.active_layer) ? 'X' : '';
			}
			file.dirty = false;
			displayActiveScale();
			drawObjects();
    } else if ((read_data.version) && (read_data.version.startsWith("2.3"))) {
			// the file is from the python version of plotter
			convertFromOld(read_data);
			file.dirty = false;
			displayActiveScale();
			drawObjects();
		}
  };
  reader.readAsText(event.target.files[0]);
  filename = event.target.files[0].name;
	document.title = `Mk. 3 Plotter (${filename})`;
}

function handleSaveButton(event) {
  let f = prompt("Enter a file name to save data:", filename);
  if (f) {
    filename = f;
		document.title = `Mk. 3 Plotter (${filename})`;
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);

    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
		file.dirty = false;
  }
}

function handleUnload(event) {
	event.preventDefault();
}

/*
 *  Functions
 */
function drawObjects() {
	ctx.clearRect(0, 0, canvas.width, canvas.height);
	data.objects.forEach((value, index, array) => {
    if (value.visible) {
      drawObject(value);
    }
  });

	calipers.display();
  displayCrosshairs();
}

function drawObject(obj, args) {
	if (obj.type === "point") {
		drawPoint(obj, args);
	} else if (obj.type === "line") {
		drawLine(obj, args);
	} else if (obj.type === "text") {
		drawText(obj, args);
	}
}

function drawPoint(point, args) {
	let x = 0;
	let y = 0;
	ctx.strokeStyle = PENCIL_COLOR;
	ctx.lineWidth = stroke_width + 0.2; // it looks better if the circles are a little darker than the lines

	if (args !== undefined) {
		if (args.x !== undefined) { x = args.x }
		if (args.y !== undefined) { y = args.y }
		if (args.color !== undefined) { ctx.strokeStyle = args.color}
		if (args.line_width !== undefined) {ctx.lineWidth = args.line_width}
	}

	if (point) {
    if ((point.layer >= 0) && (point.layer !== data.active_layer)) { return; } // only draw if the layer is active
		x = toAbsPos(point.rel_x);
		y = toAbsPos(point.rel_y);
	}


	ctx.beginPath();
	ctx.moveTo(x+point_radius, y);
	ctx.arc(x, y, point_radius, 0, 2*Math.PI);
	ctx.stroke();

	ctx.beginPath();
	ctx.moveTo(x+1, y);
	ctx.arc(x, y, 0, 0, 2*Math.PI);
	ctx.stroke();
}

function handleRangeInput() {
	let display = document.getElementById("caliperRange");
	display.value = Array.from(display.value).filter(c => "0123456789".includes(c)).join('');
	calipers.setRange(Number(display.value));
}

function drawLine(line, args) {
	let start_x			= 0;
	let start_y			= 0;
	let end_x				= 0;
	let end_y				= 0;
	let tick_dist		= 0;
	ctx.strokeStyle = BLACK_COLOR;
	ctx.lineWidth		= stroke_width;

	if (args !== undefined) {
		if (args.start_x !== undefined) { start_x = args.start_x; }
		if (args.start_y !== undefined) { start_y = args.start_y; }
		if (args.end_x !== undefined) { end_x = args.end_x; }
		if (args.end_y !== undefined) { end_y = args.end_y; }
		if (args.color !== undefined) { ctx.strokeStyle = args.color; }
		if (args.line_width !== undefined) { ctx.lineWidth = args.line_width; }
	}

	if (line) {
    if ((line.layer >= 0) && (line.layer !== data.active_layer)) { return; } // only draw if the layer is active
		start_x = toAbsPos(line.rel_start_x);
		start_y = toAbsPos(line.rel_start_y);
		end_x = toAbsPos(line.rel_end_x);
		end_y = toAbsPos(line.rel_end_y);
		// convert from miles to relative distance on the plot
		tick_dist = line.tick_dist / 332.19 / scale_factors[data.scales[data.active_layer]];
	}

	ctx.beginPath();
	ctx.moveTo(start_x, start_y);
	ctx.lineTo(end_x, end_y);
	ctx.stroke();

	if (tick_dist) {
		let angle = Math.atan2(line.rel_end_y - line.rel_start_y, line.rel_end_x - line.rel_start_x);
		let tick_count = Math.floor(lineLength(line) / tick_dist);
		let x = line.rel_start_x;
		let y = line.rel_start_y;
		for (let i=0; i<tick_count; i++) {
			x = x + (Math.cos(angle) * tick_dist);
			y = y + (Math.sin(angle) * tick_dist);
			ctx.beginPath();
			ctx.moveTo(toAbsPos(x), toAbsPos(y));
			ctx.lineTo(toAbsPos(x) + Math.cos(angle-Math.PI/2) * 5, toAbsPos(y) + Math.sin(angle-Math.PI/2) * 5);
			ctx.stroke();
		}
	}
}

function drawText(text, args) {
	let str = "";
	let x = 0;
	let y = 0;
	ctx.font = "2vh GelPenLight";
	ctx.fillStyle = PENCIL_COLOR;

	if (args !== undefined) {
		if (args.str !== undefined) { str = args.str; }
		if (args.x !== undefined) { x = args.x; }
		if (args.y !== undefined) { y = args.y; }
		if (args.font !== undefined) { ctx.font = args.font; }
		if (args.color !== undefined) { ctx.fillStyle = args.color; }
	}

	if (text) {
    if ((text.layer >= 0) && (text.layer !== data.active_layer)) { return; } // only draw if the layer is active
		str = text.text;
		x = toAbsPos(text.rel_x);
		y = toAbsPos(text.rel_y);
	}
	ctx.textBaseline = "middle";
	ctx.fillText(str, x, y);
}

// change absolute coordinate to relative to size of window
function toRelPos(x) {
  return x / document.documentElement.clientHeight;
}

// change coordinate relative to window size to absoulte pixel location
function toAbsPos(x) {
  return x * document.documentElement.clientHeight;
}

// distance from a point to a point
// if p is specified it is a Point class and used for one of the points
// if a_x and a_y are omitted 0.5 is used which is the relative center of the graph
function distanceToPoint(p, a_x=0.5, a_y=0.5, b_x, b_y) {
  if (p) {
    b_x = p.rel_x;
    b_y = p.rel_y;
  }
  return Math.sqrt(Math.pow(a_x - b_x, 2) + Math.pow(a_y - b_y, 2));
}

// distance from a line to a point
// if l is specified it is a Line class and used for start and end of the line
// if p_x and p_y are omitted 0.5 is used which is the relative center of the graph
function distanceToLine(l, p_x=0.5, p_y=0.5, s_x, s_y, e_x, e_y) {
  if (l) {
    s_x = l.rel_start_x;
    s_y = l.rel_start_y;
    e_x = l.rel_end_x;
    e_y = l.rel_end_y;
  }

  let AB = [];
	AB[0] = e_x - s_x;
	AB[1] = e_y - s_y;

	let BE = [];
	BE[0] = p_x - e_x;
	BE[1] = p_y - e_y;

	let AE = [];
	AE[0] = p_x - s_x;
	AE[1] = p_y - s_y;

	// Calculating the dot product
	let AB_BE = (AB[0] * BE[0]) + (AB[1] * BE[1]);
	let AB_AE = (AB[0] * AE[0]) + (AB[1] * AE[1]);

	let result = 0;

	if (AB_BE > 0) {
		let y = p_y - e_y;
		let x = p_x - e_x;
		result = Math.sqrt((x * x) + (y * y));
  } else if (AB_AE < 0) {
		let y = p_y - s_y;
		let x = p_x - s_x;
		result = Math.sqrt((x * x) + (y * y));
  } else {
		let x1 = AB[0];
		let y1 = AB[1];
		let x2 = AE[0];
		let y2 = AE[1];
		let mod = Math.sqrt((x1 * x1) + (y1 * y1));
		result = Math.abs((x1 * y2) - (y1 * x2)) / mod;
  }

	return result;
}

// returns distance from a Text class to a point
// it uses the a line across the center of the text as the reference for distance
// if point is not specified 0.5 is used which is the relative center of the graph
function distanceToText(t, p_x=0.5, p_y=0.5) {
	const s_x = t.rel_x;
	const s_y = t.rel_y;
	const e_x = t.rel_x + toRelPos(ctx.measureText(t.text).width);
	const e_y = t.rel_y;
	return distanceToLine(null, p_x, p_y, s_x, s_y, e_x, e_y);
}

function changeActiveScale() {
  data.scales[data.active_layer] += 1;
  data.scales[data.active_layer] %= 3;

  let scale_factor = null;
  switch (data.scales[data.active_layer]) {
    case 0:
      scale_factor = 300 / 150;
      break;

    case 1:
      scale_factor = 150 / 200;
      break;

    case 2:
      scale_factor = 200 / 300;
      break;
  }

  data.objects.forEach((value, index, array) => {
		if (value.type === "point") {
			if (value.layer === data.active_layer) {
				value.rel_x = ((value.rel_x - 0.5) * scale_factor) + 0.5;
				value.rel_y = ((value.rel_y - 0.5) * scale_factor) + 0.5;
			}
		} else if (value.type === "line") {
			if (value.layer === data.active_layer) {
				value.rel_start_x = ((value.rel_start_x - 0.5) * scale_factor) + 0.5;
				value.rel_start_y = ((value.rel_start_y - 0.5) * scale_factor) + 0.5;
				value.rel_end_x = ((value.rel_end_x - 0.5) * scale_factor) + 0.5;
				value.rel_end_y = ((value.rel_end_y - 0.5) * scale_factor) + 0.5;
			}
		} else if (value.type === "text") {
			if (value.layer === data.active_layer) {
				value.rel_x = ((value.rel_x - 0.5) * scale_factor) + 0.5;
				value.rel_y = ((value.rel_y - 0.5) * scale_factor) + 0.5;
			}
		}
  });
}

function displayActiveScale() {
  // change the scale
	for (let i=0; i<3; i++) {
		if (data.scales[data.active_layer] === i) {
			graph_elements[i].style.display = "block";
			if (displayActiveScale.display_map[data.active_layer]) {
				map_elements[i].style.display = "block";
			} else {
				map_elements[i].style.display = "none";
			}
		} else {
			graph_elements[i].style.display = "none";
			map_elements[i].style.display = "none";
		}
	}
}

// display a vertical and horizontal line to use as a crosshair
function displayCrosshairs() {
  if (displayCrosshairs.show) {
    if (toRelPos(global_mouse.x) < 1) {
      drawLine(null, {
				start_x:		global_mouse.x,
				start_y:		0,
				end_x:			global_mouse.x,
				end_y:			toAbsPos(1),
				line_width:	0.4,
			});
      drawLine(null, {
				start_x:		0,
				start_y:		global_mouse.y,
				end_x:			toAbsPos(1),
				end_y:			global_mouse.y,
				line_width:	0.4,
			});
    }
  }
}

function rotatePoint(x, y, a) {
	let r_x = (((x - 0.5) * Math.cos(a)) - ((y - 0.5) * Math.sin(a))) + 0.5;
	let r_y = (((x - 0.5) * Math.sin(a)) + ((y - 0.5) * Math.cos(a))) + 0.5;
	return [r_x, r_y];
}

// selectet the two closet objects to the cursor and stores them in the 'sel' global variable
function selectClosest() {
  selected.reset();
	let close_distance = 1000;
	let next_distance = 1000;
  data.objects.forEach(function(value, index, array) {
    if ((value.layer < 0) || (value.layer === data.active_layer)) {
			let d = null;
			if (value.type === "point") {
				d = distanceToPoint(value, toRelPos(global_mouse.x), toRelPos(global_mouse.y));
			} else if (value.type === "line") {
				d = distanceToLine(value, toRelPos(global_mouse.x), toRelPos(global_mouse.y));
			} else if (value.type === "text") {
				d = distanceToText(value, toRelPos(global_mouse.x), toRelPos(global_mouse.y));
			}
      if (d < close_distance) {
				next_distance = close_distance;
        close_distance = d;

				selected.object_index[1] = selected.object_index[0];
				selected.object_index[0] = index;
      } else if (d < next_distance) {
				next_distance = d;
				selected.object_index[1] = index;
			}
    }
  });

	if (selected.object_index[0] !== null) {
		// something was found
    drawObjects();
		drawObject(data.objects[selected.object_index[0]], { color:RED_COLOR, line_width:stroke_width+0.4, });
	} else {
		// there are no visible items
		return false;
	}

  // something was found
  return true;
}

// find the length of a line object
function lineLength(l) {
	let a = l.rel_end_x - l.rel_start_x;
	let b = l.rel_end_y - l.rel_start_y;
	return Math.sqrt((a*a) + (b*b));
}

function convertFromOld(read_data) {
	data.scales.splice(0, data.scales.length);
	data.objects.splice(0, data.objects.length);

	data.active_layer = read_data.active_layer - 1;

	for (let i=0; i<LAYER_COUNT; i++) {
		switch (read_data.scales[i+1]) {
			case 0:
				data.scales[i] = 2;
				break;

			case 1:
				data.scales[i] = 0;
				break;

			case 2:
				data.scales[i] = 1;
				break;
		}
	}

	read_data.items.forEach((value, index, array) => {
		switch (value.TYPE) {
			case 0: // point
				data.objects.push(new Point(convertPos(value.DRAW_LOC[0]), convertPos(value.DRAW_LOC[1])));
				data.objects.at(-1).layer = (value.LAYER === 0) ? -1 : value.LAYER - 1;
				break;

			case 1: // line
				data.objects.push(new Line(
					convertPos(value.START_LOC[0]),
					convertPos(value.START_LOC[1]),
					convertPos(value.END_LOC[0]),
					convertPos(value.END_LOC[1]),
				));
				data.objects.at(-1).layer = (value.LAYER === 0) ? -1 : value.LAYER - 1;
				break;

			case 2: // text
				data.objects.push(new Text(
					value.TEXT_DATA,
					convertPos(value.DRAW_LOC[0]),
					convertPos(value.DRAW_LOC[1]) + convertPos(24),
				));
				data.objects.at(-1).layer = (value.LAYER === 0) ? -1 : value.LAYER - 1;
				data.objects.at(-1).clean = false;
				break;
		}
	});
}

function convertPos(n) {
	return toAbsPos(n / 900);
}
