-----------------------------------------------------------------
-- Mortar Team System (MOOSE) — Static Slots
-- Version: v2.3
-- Description:
--   Static F10 menus (no refresh needed): Mortar Team 1/2/3, each with:
--   Get / Load / Unload / Extract / Rearm / Request Mark / Status + global Field Status + RTB.
--   Timers: Get/Load/Unload/Extract = 10s; Rearm = 20s.
--   Rearm pool: 2 rearms per team; needs a helo within 200 m to reset after depletion.
--   Unload offsets by slot: Team1=20m, Team2=40m, Team3=60m behind helo.
--   Smoke request: 100 m forward along team heading.
--   Map marker: 500 m to the RIGHT of team heading; visible to ALL coalitions, READ-ONLY.
--
-- Author: Aidi 
-- Date: 21-Oct-2025
-----------------------------------------------------------------

_SETTINGS:SetPlayerMenuOff()

-----------------------------------------------------------------
-- CONFIG
-----------------------------------------------------------------
local TEAM_SLOT_COUNT        = 3
local TEAM_NAMES_POOL        = { "Guardian", "Vanguard", "Auxilio", "Vitalis", "Falcon", "Valour" }
local MAX_DEPLOYED_TEAMS     = 3            -- world limit
local MAX_EMBARKED_PER_HELO  = 3            -- per helo
local PROXIMITY_M            = 200          -- for Load / Extract / Rearm and helo resupply
local LOADZONE_NAME          = "Loadzone"   -- must exist in ME
local MARK_RADIUS_M          = 3000         -- request mark gating
local SMOKE_OFFSET_FWD_M     = 100          -- smoke 100m forward
local GET_SIDE_OFFSET_M      = 12           -- 12 m right of helo for GET
local UNLOAD_SLOT_OFFSETS_M  = { 20, 40, 60 } -- by slot id (1..3), behind helo
local UNLOAD_DELAY_SECONDS   = 10
local ACTION_DELAY_SECONDS   = 10           -- Get / Load / Extract / Unload delays
local REARM_DELAY_SECONDS    = 20           -- Rearm delay
local MARKER_OFFSET_RIGHT_M  = 500          -- map marker offset to the RIGHT of team heading
local MORTAR_TEMPLATE        = "Mortar"     -- ME template (Late Activation)
local PERSIST_FILE           = "C:/Users/Aidi/Saved Games/DCS/Missions/Saves/Kiowa_Training/Saves/MortarTeams_static.lua"
local REARMS_PER_POOL        = 2            -- rearm attempts before needing helo resupply

-- Optional SRS screen mirror (safe if MSRS not present)
MIRROR_SRS_TO_SCREEN = MIRROR_SRS_TO_SCREEN or false
SRS_POPUP_PREFIX     = SRS_POPUP_PREFIX or "Mortar: "
SRS_ROLES = SRS_ROLES or {}

-----------------------------------------------------------------
-- STATE (single source of truth)
-----------------------------------------------------------------
-- Teams[i] = { slot=i, name, state="EMPTY"|"DEPLOYED"|"EMBARKED",
--              coord=COORDINATE|nil, hdg=deg|0, embarkedBy=heloName|nil, groups={...},
--              rearmsRemaining=int }
local Teams = {}
local UsedNameBase = {}   -- base name usage tracker
local TeamLocks = {}      -- soft locks per slot (unlock time, seconds)

-- Embarked list per helo (for capacity checks)
-- EmbarkedByHelo[heloName] = { "Guardian Team", ... }
local EmbarkedByHelo = {}

-- Map markers per team slot (DCS marker IDs)
local TeamMarkers = {}
local MARKER_ID_SEQ = 9000 -- starting id; must be unique and numeric

-- Persistent SPAWNER (prevents name collisions)
local MORTAR_SPAWNER = nil

-----------------------------------------------------------------
-- UTILITIES
-----------------------------------------------------------------
local function _popup(t, s)
  if _G.MESSAGE and MESSAGE.New then MESSAGE:New(t, s or 8):ToAll()
  else trigger.action.outText(t, s or 8) end
end

local function SRS_Say(role, text, show)
  if not text or text=="" then return end
  local doMirror = (show==true) or (show==nil and MIRROR_SRS_TO_SCREEN)
  if doMirror then _popup((SRS_POPUP_PREFIX or "")..text, 8) end
end

local function serialize(val, indent)
  indent = indent or ""
  local t = type(val)
  if t=="number" or t=="boolean" then return tostring(val)
  elseif t=="string" then return string.format("%q", val)
  elseif t=="table" then
    local ni=indent.."  "; local out={"{\n"}
    for k,v in pairs(val) do
      local key=(type(k)=="string" and k:match("^%a[%w_]*$")) and k or "["..serialize(k,ni).."]"
      out[#out+1] = ni..key.." = "..serialize(v,ni)..",\n"
    end
    out[#out+1] = indent.."}"; return table.concat(out)
  end
  return "nil"
end

local function ensureDir(path)
  local dir = path:match("^(.*)/[^/]+$"); if not dir then return end
  local seg=""; for part in dir:gmatch("[^/]+") do seg=(seg=="" and part or (seg.."/"..part)); pcall(lfs.mkdir, seg) end
end

local function GetPlayerHelo()
  local set = SET_GROUP:New():FilterCoalitions("blue"):FilterCategoryHelicopter():FilterActive(true):FilterStart()
  local playerU
  set:ForEachGroup(function(g)
    if playerU then return end
    local u = g:GetUnit(1)
    if u and u:IsAlive() and u:IsPlayer() then playerU = u end
  end)
  return playerU
end

local function _isInLoadzone(coord)
  local z = trigger.misc.getZone(LOADZONE_NAME)
  if not (z and z.point and z.radius) then return false end
  local p = coord:GetVec3()
  local dx, dz = p.x - z.point.x, p.z - z.point.z
  return (dx*dx + dz*dz) <= (z.radius*z.radius)
end

local function _degHeading(h)
  if not h then return 0 end
  if h > 2*math.pi then return h % 360 end
  return (math.deg(h)) % 360
end

local function _countDeployed()
  local n=0; for i=1,TEAM_SLOT_COUNT do if Teams[i].state=="DEPLOYED" then n=n+1 end end
  return n
end

local function _embarkedCount(heloName)
  local t = EmbarkedByHelo[heloName] or {}; return #t
end

local function _embarkedHas(heloName, teamName)
  local t = EmbarkedByHelo[heloName] or {}
  for _,n in ipairs(t) do if n==teamName then return true end end
  return false
end

local function _embarkedAdd(heloName, teamName)
  EmbarkedByHelo[heloName] = EmbarkedByHelo[heloName] or {}
  local t = EmbarkedByHelo[heloName]
  if #t >= MAX_EMBARKED_PER_HELO then return false, "Capacity reached" end
  if _embarkedHas(heloName, teamName) then return false, "Already embarked" end
  t[#t+1] = teamName; return true
end

local function _embarkedRemove(heloName, teamName)
  local t = EmbarkedByHelo[heloName]; if not t then return end
  for i,n in ipairs(t) do if n==teamName then table.remove(t,i); return end end
end

local function _pickTeamNameForSlot(slot)
  for _,base in ipairs(TEAM_NAMES_POOL) do
    if not UsedNameBase[base] then UsedNameBase[base]=true; return base.." Team" end
  end
  return string.format("Mortar %d", slot)
end

local function _destroyGroups(list)
  if not list then return end
  for _,gname in ipairs(list) do
    local g = Group.getByName(gname)
    if g and g.destroy then pcall(function() g:destroy() end) end
  end
end

local function _spawnTeamAt(slot, coord, hdg)
  if not SPAWN or not MORTAR_SPAWNER then
    MESSAGE:New("MOOSE SPAWN not available.",8):ToCoalition(coalition.side.BLUE); return false
  end
  MORTAR_SPAWNER:InitHeading(hdg)
  local grp = MORTAR_SPAWNER:SpawnFromCoordinate(coord)
  Teams[slot].groups = {}
  if grp and grp.GetName then Teams[slot].groups[1] = grp:GetName() end
  return true
end

-----------------------------------------------------------------
-- MAP MARKERS
-----------------------------------------------------------------
local function _newMarkerId() MARKER_ID_SEQ = MARKER_ID_SEQ + 1; return MARKER_ID_SEQ end

local function _createMarkerForSlot(slot)
  -- place marker 500 m to the RIGHT of team heading, visible to ALL coalitions, READ-ONLY
  local t = Teams[slot]
  if not (t and t.state=="DEPLOYED" and t.coord) then return end
  local hdg = (t.hdg or 0) + 90
  local markerCoord = t.coord:Translate(MARKER_OFFSET_RIGHT_M, hdg)
  local id = _newMarkerId()
  local text = string.format("Mortar Team %d (%s) — Deployed", slot, t.name or "Mortar")
  trigger.action.markToAll(id, text, markerCoord:GetVec3(), true) -- readOnly=true
  TeamMarkers[slot] = id
end

local function _removeMarkerForSlot(slot)
  local id = TeamMarkers[slot]
  if id then trigger.action.removeMark(id); TeamMarkers[slot] = nil end
end

-----------------------------------------------------------------
-- PERSISTENCE
-----------------------------------------------------------------
local function SaveState()
  ensureDir(PERSIST_FILE)
  local save = {}
  for i=1,TEAM_SLOT_COUNT do
    local t = Teams[i]
    local v = t.coord and t.coord:GetVec3() or nil
    save[#save+1] = {
      slot = i,
      name = t.name,
      state = t.state,
      hdg = t.hdg or 0,
      embarkedBy = t.embarkedBy,
      rearmsRemaining = t.rearmsRemaining or REARMS_PER_POOL,
      coord = v and {x=v.x, y=v.y or 0, z=v.z} or nil
    }
  end
  local f = io.open(PERSIST_FILE,"w"); if not f then return end
  f:write("return ", serialize(save), "\n"); f:close()
end

local function LoadState()
  local f = io.open(PERSIST_FILE,"r"); if not f then return end
  local chunk = f:read("*a"); f:close()
  local ok, data = pcall(loadstring(chunk)); if not ok or type(data)~="table" then return end
  for _,rec in ipairs(data) do
    local i = tonumber(rec.slot or 0)
    if i and Teams[i] then
      Teams[i].name = rec.name or Teams[i].name
      Teams[i].state = rec.state or "EMPTY"
      Teams[i].hdg = rec.hdg or 0
      Teams[i].embarkedBy = rec.embarkedBy
      Teams[i].rearmsRemaining = tonumber(rec.rearmsRemaining or REARMS_PER_POOL)
      Teams[i].groups = {}
      if rec.coord then
        Teams[i].coord = COORDINATE:NewFromVec3({x=rec.coord.x, y=rec.coord.y or 0, z=rec.coord.z})
        if Teams[i].state=="DEPLOYED" then
          if _spawnTeamAt(i, Teams[i].coord, Teams[i].hdg or 0) then
            _createMarkerForSlot(i)
          end
        end
      else
        Teams[i].coord = nil
      end
    end
  end
end

-----------------------------------------------------------------
-- VALIDATION / LOCKS
-----------------------------------------------------------------
local function _locked(slot)
  local untilT = TeamLocks[slot]
  return untilT and timer.getTime() < untilT
end

local function _lock(slot, secs)
  TeamLocks[slot] = timer.getTime() + (secs or 5)
end

-----------------------------------------------------------------
-- ACTIONS (per-slot)
-----------------------------------------------------------------
local function Action_Get(slot)
  if _locked(slot) then MESSAGE:New("Team "..slot.." busy. Try again.",7):ToCoalition(coalition.side.BLUE); return end
  local helo = GetPlayerHelo(); if not helo then MESSAGE:New("No player helo active.",8):ToCoalition(coalition.side.BLUE); return end
  local hc = helo:GetCoordinate()
  if not _isInLoadzone(hc) then MESSAGE:New("Enter the Loadzone to form Team "..slot..".",8):ToCoalition(coalition.side.BLUE); return end
  if Teams[slot].state~="EMPTY" then MESSAGE:New("Team "..slot.." already formed.",8):ToCoalition(coalition.side.BLUE); return end
  if _countDeployed()>=MAX_DEPLOYED_TEAMS then MESSAGE:New("Max deployed teams reached ("..MAX_DEPLOYED_TEAMS..").",8):ToCoalition(coalition.side.BLUE); return end

  local hdg = _degHeading(helo:GetHeading())
  local behind = UNLOAD_SLOT_OFFSETS_M[slot] or 20
  local spawn = hc:Translate(behind, (hdg+180)%360):Translate(GET_SIDE_OFFSET_M, (hdg+90)%360)

  if not Teams[slot].name or Teams[slot].name=="" then Teams[slot].name = _pickTeamNameForSlot(slot) end
  if Teams[slot].rearmsRemaining == nil then Teams[slot].rearmsRemaining = REARMS_PER_POOL end

  MESSAGE:New(string.format("Forming Team %d (%s)... %ds", slot, Teams[slot].name, ACTION_DELAY_SECONDS),8):ToCoalition(coalition.side.BLUE)
  _lock(slot, ACTION_DELAY_SECONDS)

  SCHEDULER:New(nil, function()
    if _spawnTeamAt(slot, spawn, hdg) then
      Teams[slot].state="DEPLOYED"; Teams[slot].coord=spawn; Teams[slot].hdg=hdg; Teams[slot].embarkedBy=nil
      _createMarkerForSlot(slot)
      MESSAGE:New(string.format("Team %d (%s) formed.", slot, Teams[slot].name), 8):ToCoalition(coalition.side.BLUE)
      SRS_Say("Mortar", string.format("Team %d formed.", slot), false)
      SaveState()
    end
  end, {}, ACTION_DELAY_SECONDS)
end

local function Action_Load(slot)
  if _locked(slot) then MESSAGE:New("Team "..slot.." busy. Try again.",7):ToCoalition(coalition.side.BLUE); return end
  local helo = GetPlayerHelo(); if not helo then MESSAGE:New("No player helo active.",8):ToCoalition(coalition.side.BLUE); return end
  local hc = helo:GetCoordinate()
  local uname = helo:GetName() or "helo"
  if Teams[slot].state~="DEPLOYED" or not Teams[slot].coord then MESSAGE:New("Team "..slot.." not deployed.",8):ToCoalition(coalition.side.BLUE); return end
  local dist = hc:Get2DDistance(Teams[slot].coord)
  if dist>PROXIMITY_M then MESSAGE:New(string.format("Move within %dm to load Team %d.", PROXIMITY_M, slot),8):ToCoalition(coalition.side.BLUE); return end
  if _embarkedCount(uname)>=MAX_EMBARKED_PER_HELO then MESSAGE:New("Helo capacity reached.",8):ToCoalition(coalition.side.BLUE); return end

  MESSAGE:New(string.format("Loading Team %d (%s)... %ds", slot, Teams[slot].name, ACTION_DELAY_SECONDS),8):ToCoalition(coalition.side.BLUE)
  _lock(slot, ACTION_DELAY_SECONDS)

  SCHEDULER:New(nil, function()
    _destroyGroups(Teams[slot].groups); Teams[slot].groups={}
    _removeMarkerForSlot(slot)
    local ok,why = _embarkedAdd(uname, Teams[slot].name)
    if not ok then MESSAGE:New("Cannot load: "..tostring(why),8):ToCoalition(coalition.side.BLUE); return end
    Teams[slot].state="EMBARKED"; Teams[slot].embarkedBy=uname; Teams[slot].coord=nil
    MESSAGE:New(string.format("Team %d (%s) embarked.", slot, Teams[slot].name),8):ToCoalition(coalition.side.BLUE)
    SaveState()
  end, {}, ACTION_DELAY_SECONDS)
end

local function Action_Extract(slot)
  -- same preconditions as Load
  Action_Load(slot)
end

local function Action_Unload(slot)
  if _locked(slot) then MESSAGE:New("Team "..slot.." busy. Try again.",7):ToCoalition(coalition.side.BLUE); return end
  local helo = GetPlayerHelo(); if not helo then MESSAGE:New("No player helo active.",8):ToCoalition(coalition.side.BLUE); return end
  local uname = helo:GetName() or "helo"
  if Teams[slot].state~="EMBARKED" or Teams[slot].embarkedBy~=uname then
    MESSAGE:New("Team "..slot.." not embarked on your helo.",8):ToCoalition(coalition.side.BLUE); return
  end
  if _countDeployed()>=MAX_DEPLOYED_TEAMS then MESSAGE:New("Max deployed teams reached ("..MAX_DEPLOYED_TEAMS..").",8):ToCoalition(coalition.side.BLUE); return end

  MESSAGE:New(string.format("Deploying Team %d in %ds...", slot, ACTION_DELAY_SECONDS),8):ToCoalition(coalition.side.BLUE)
  _lock(slot, ACTION_DELAY_SECONDS)

  SCHEDULER:New(nil, function()
    local hc = helo:GetCoordinate()
    local hdg = _degHeading(helo:GetHeading())
    local offset = UNLOAD_SLOT_OFFSETS_M[slot] or 20
    local spawn = hc:Translate(offset, (hdg+180)%360)

    if _spawnTeamAt(slot, spawn, hdg) then
      Teams[slot].state="DEPLOYED"; Teams[slot].coord=spawn; Teams[slot].hdg=hdg; Teams[slot].embarkedBy=nil
      _embarkedRemove(uname, Teams[slot].name)
      _createMarkerForSlot(slot)
      MESSAGE:New(string.format("Team %d (%s) deployed.", slot, Teams[slot].name),8):ToCoalition(coalition.side.BLUE)
      SaveState()
    end
  end, {}, ACTION_DELAY_SECONDS)
end

local function Action_Rearm(slot)
  if _locked(slot) then MESSAGE:New("Team "..slot.." busy. Try again.",7):ToCoalition(coalition.side.BLUE); return end
  local helo = GetPlayerHelo(); if not helo then MESSAGE:New("No player helo active.",8):ToCoalition(coalition.side.BLUE); return end
  local t = Teams[slot]
  if t.state~="DEPLOYED" or not t.coord then MESSAGE:New("Team "..slot.." not deployed.",8):ToCoalition(coalition.side.BLUE); return end

  local hc = helo:GetCoordinate()
  local dist = hc:Get2DDistance(t.coord)

  -- Rearm pool logic
  if t.rearmsRemaining == nil then t.rearmsRemaining = REARMS_PER_POOL end

  if t.rearmsRemaining <= 0 then
    -- Need helicopter within PROXIMITY_M to reset pool
    if dist > PROXIMITY_M then
      MESSAGE:New(string.format("Team %d is out of ammo. Bring a helicopter within %dm to resupply.", slot, PROXIMITY_M), 10):ToCoalition(coalition.side.BLUE)
      return
    else
      t.rearmsRemaining = REARMS_PER_POOL
      MESSAGE:New(string.format("Team %d resupplied by helicopter. Rearms available: %d.", slot, t.rearmsRemaining), 10):ToCoalition(coalition.side.BLUE)
    end
  end

  -- Must still be within PROXIMITY_M to execute rearm
  if dist > PROXIMITY_M then
    MESSAGE:New(string.format("Move within %dm to rearm Team %d.", PROXIMITY_M, slot),8):ToCoalition(coalition.side.BLUE)
    return
  end

  -- Consume one rearm and simulate the 20s rearm process
  t.rearmsRemaining = math.max(0, (t.rearmsRemaining or 0) - 1)
  MESSAGE:New(string.format("Rearming Team %d... %ds (rearms left after: %d)", slot, REARM_DELAY_SECONDS, t.rearmsRemaining), 10):ToCoalition(coalition.side.BLUE)

  _lock(slot, REARM_DELAY_SECONDS)
  SCHEDULER:New(nil, function()
    -- despawn/respawn to simulate fresh ammo
    _destroyGroups(t.groups); t.groups={}
    if _spawnTeamAt(slot, t.coord, t.hdg or 0) then
      MESSAGE:New(string.format("Team %d rearmed. Rearms remaining: %d", slot, t.rearmsRemaining),8):ToCoalition(coalition.side.BLUE)
      SaveState()
    end
  end, {}, REARM_DELAY_SECONDS)
end

-- Request Mark: Smoke 100 m forward along team heading
local function Action_RequestSmoke(slot)
  local t = Teams[slot]
  if t.state~="DEPLOYED" or not t.coord then MESSAGE:New("Team "..slot.." not deployed.",8):ToCoalition(coalition.side.BLUE); return end
  local helo = GetPlayerHelo(); if not helo then return end
  local hc = helo:GetCoordinate()
  local dist = hc:Get2DDistance(t.coord)
  if dist>MARK_RADIUS_M then
    MESSAGE:New(string.format("Be within %.1f km to request smoke (now %.2f km).", MARK_RADIUS_M/1000, dist/1000), 10):ToCoalition(coalition.side.BLUE); return
  end
  local hdg = t.hdg or 0
  local smokeCoord = t.coord:Translate(SMOKE_OFFSET_FWD_M, hdg)
  trigger.action.smoke(smokeCoord:GetVec3(), trigger.smokeColor.Green)
  MESSAGE:New(string.format("Smoke deployed ~%dm forward of Team %d.", SMOKE_OFFSET_FWD_M, slot),8):ToCoalition(coalition.side.BLUE)
end

local function Action_RequestFlare(slot)
  local t = Teams[slot]
  if t.state~="DEPLOYED" or not t.coord then MESSAGE:New("Team "..slot.." not deployed.",8):ToCoalition(coalition.side.BLUE); return end
  local helo = GetPlayerHelo(); if not helo then return end
  local hc = helo:GetCoordinate()
  local dist = hc:Get2DDistance(t.coord)
  if dist>MARK_RADIUS_M then
    MESSAGE:New(string.format("Be within %.1f km to request a flare (now %.2f km).", MARK_RADIUS_M/1000, dist/1000), 10):ToCoalition(coalition.side.BLUE); return
  end
  trigger.action.signalFlare(t.coord:GetVec3(), trigger.flareColor.Green, 0)
  MESSAGE:New(string.format("Flare at Team %d.", slot),8):ToCoalition(coalition.side.BLUE)
end

local function Action_Status(slot)
  local helo = GetPlayerHelo()
  local hc = helo and helo:GetCoordinate() or nil
  local t = Teams[slot]
  local line
  local ammoTxt = string.format(" (rearms left: %d)", t.rearmsRemaining or REARMS_PER_POOL)
  if t.state=="EMPTY" then
    line = string.format("Team %d: EMPTY%s", slot, ammoTxt)
  elseif t.state=="EMBARKED" then
    line = string.format("Team %d (%s): EMBARKED by %s%s", slot, t.name or "Mortar", t.embarkedBy or "unknown", ammoTxt)
  elseif t.state=="DEPLOYED" and t.coord then
    if hc then
      local d = hc:Get2DDistance(t.coord)
      local brg = math.floor(hc:HeadingTo(t.coord)+0.5)
      line = string.format("Team %d (%s): DEPLOYED — %.2f km @ %03d°%s", slot, t.name, d/1000, brg, ammoTxt)
    else
      line = string.format("Team %d (%s): DEPLOYED%s", slot, t.name, ammoTxt)
    end
  else
    line = string.format("Team %d: state unknown%s", slot, ammoTxt)
  end
  MESSAGE:New(line, 10):ToCoalition(coalition.side.BLUE)
  SRS_Say("Mortar", line, false)
end

local function Action_FieldStatus()
  local helo = GetPlayerHelo()
  local hc = helo and helo:GetCoordinate() or nil
  local lines = {"Mortar Field Status:"}
  for i=1,TEAM_SLOT_COUNT do
    local t = Teams[i]
    local ammoTxt = string.format(" (rearms left: %d)", t.rearmsRemaining or REARMS_PER_POOL)
    if t.state=="EMPTY" then
      lines[#lines+1] = string.format("  • Team %d: EMPTY%s", i, ammoTxt)
    elseif t.state=="EMBARKED" then
      lines[#lines+1] = string.format("  • Team %d (%s): EMBARKED by %s%s", i, t.name or "Mortar", t.embarkedBy or "unknown", ammoTxt)
    elseif t.state=="DEPLOYED" and t.coord then
      if hc then
        local d = hc:Get2DDistance(t.coord)
        local brg = math.floor(hc:HeadingTo(t.coord)+0.5)
        lines[#lines+1] = string.format("  • Team %d (%s): DEPLOYED — %.2f km @ %03d°%s", i, t.name or "Mortar", d/1000, brg, ammoTxt)
      else
        lines[#lines+1] = string.format("  • Team %d (%s): DEPLOYED%s", i, t.name or "Mortar", ammoTxt)
      end
    else
      lines[#lines+1] = string.format("  • Team %d: state unknown%s", i, ammoTxt)
    end
  end
  local txt = table.concat(lines, "\n")
  MESSAGE:New(txt, 12):ToCoalition(coalition.side.BLUE)
  SRS_Say("Mortar", "Field status updated.", false)
end

-- RTB (Reset Team) — reset a team to EMPTY, remove unit/marker/embark, reset ammo pool
local function Action_RTB(slot)
  if _locked(slot) then MESSAGE:New("Team "..slot.." busy. Try again.",7):ToCoalition(coalition.side.BLUE); return end
  _lock(slot, 3)
  local t = Teams[slot]

  -- If embarked, clear from that helo
  if t.state=="EMBARKED" and t.embarkedBy then
    _embarkedRemove(t.embarkedBy, t.name or ("Team "..slot))
  end

  -- If deployed, destroy group & remove marker
  if t.state=="DEPLOYED" then
    _destroyGroups(t.groups); t.groups={}
    _removeMarkerForSlot(slot)
  end

  -- Reset slot
  t.state="EMPTY"; t.coord=nil; t.hdg=0; t.embarkedBy=nil
  t.rearmsRemaining = REARMS_PER_POOL
  MESSAGE:New(string.format("Team %d reset to base (EMPTY). Rearms reset to %d.", slot, REARMS_PER_POOL),8):ToCoalition(coalition.side.BLUE)
  SaveState()
end

-----------------------------------------------------------------
-- MENUS (STATIC)
-----------------------------------------------------------------
local Root = MENU_COALITION:New(coalition.side.BLUE, "Mortar Team")

local function BuildTeamMenu(slot)
  local label = string.format("Mortar Team %d", slot)
  local m = MENU_COALITION:New(coalition.side.BLUE, label, Root)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Get",           m, function() Action_Get(slot) end)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Load",          m, function() Action_Load(slot) end)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Unload",        m, function() Action_Unload(slot) end)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Extract",       m, function() Action_Extract(slot) end)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Rearm (20s)",   m, function() Action_Rearm(slot) end)
  local mark = MENU_COALITION:New(coalition.side.BLUE, "Request Mark", m)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Smoke (100 m fwd)", mark, function() Action_RequestSmoke(slot) end)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Flare (at team)",   mark, function() Action_RequestFlare(slot) end)
  MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Status",        m, function() Action_Status(slot) end)
end

for i=1,TEAM_SLOT_COUNT do
  Teams[i]={slot=i, name="", state="EMPTY", coord=nil, hdg=0, embarkedBy=nil, groups={}, rearmsRemaining=REARMS_PER_POOL}
  BuildTeamMenu(i)
end

-- Global items
MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Field Mortar Status", Root, Action_FieldStatus)

-- RTB Menu
local RTBMenu = MENU_COALITION:New(coalition.side.BLUE, "RTB (Reset Team)", Root)
MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Team 1 → RTB", RTBMenu, function() Action_RTB(1) end)
MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Team 2 → RTB", RTBMenu, function() Action_RTB(2) end)
MENU_COALITION_COMMAND:New(coalition.side.BLUE, "Team 3 → RTB", RTBMenu, function() Action_RTB(3) end)

-----------------------------------------------------------------
-- INIT / PERSISTENCE
-----------------------------------------------------------------
if SPAWN then
  MORTAR_SPAWNER = SPAWN:New(MORTAR_TEMPLATE)
else
  env.info("MOOSE SPAWN missing; spawns will fail.")
end

-- Load any previous state, respawn deployed teams, and restore markers
SCHEDULER:New(nil, LoadState, {}, 1)

env.info("Mortar Team System (Static) v2.3 by Aidi + ChatGPT (GPT-5) loaded")

-----------------------------------------------------------------
-- Mortar Team System (Static) v2.3 by Aidi
-- Baseline release finalized: 21-Oct-2025
-----------------------------------------------------------------
